/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.model.loading;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;

import com.google.common.collect.ImmutableList;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.Reference2ReferenceMap;
import it.unimi.dsi.fastutil.objects.Reference2ReferenceOpenHashMap;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.client.model.loading.v1.BlockStateResolver;
import net.fabricmc.fabric.api.client.model.loading.v1.ModelLoadingPlugin;
import net.fabricmc.fabric.api.client.model.loading.v1.ModelModifier;
import net.minecraft.class_1087;
import net.minecraft.class_1091;
import net.minecraft.class_1100;
import net.minecraft.class_2248;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_3665;
import net.minecraft.class_5321;
import net.minecraft.class_773;
import net.minecraft.class_7775;
import net.minecraft.class_7923;
import net.minecraft.class_9824;
import net.minecraft.class_9979;

public class ModelLoadingEventDispatcher {
	private static final Logger LOGGER = LoggerFactory.getLogger(ModelLoadingEventDispatcher.class);
	public static final ThreadLocal<ModelLoadingEventDispatcher> CURRENT = new ThreadLocal<>();

	private final ModelLoadingPluginContextImpl pluginContext;

	private final BlockStateResolverContext blockStateResolverContext = new BlockStateResolverContext();

	private final OnLoadModifierContext onLoadModifierContext = new OnLoadModifierContext();
	private final ObjectArrayList<BeforeBakeModifierContext> beforeBakeModifierContextStack = new ObjectArrayList<>();
	private final ObjectArrayList<AfterBakeModifierContext> afterBakeModifierContextStack = new ObjectArrayList<>();

	private final OnLoadBlockModifierContext onLoadBlockModifierContext = new OnLoadBlockModifierContext();
	private final BeforeBakeBlockModifierContext beforeBakeBlockModifierContext = new BeforeBakeBlockModifierContext();
	private final AfterBakeBlockModifierContext afterBakeBlockModifierContext = new AfterBakeBlockModifierContext();

	public ModelLoadingEventDispatcher(List<ModelLoadingPlugin> plugins) {
		this.pluginContext = new ModelLoadingPluginContextImpl();

		for (ModelLoadingPlugin plugin : plugins) {
			try {
				plugin.initialize(pluginContext);
			} catch (Exception exception) {
				LOGGER.error("Failed to initialize model loading plugin", exception);
			}
		}
	}

	public void forEachExtraModel(Consumer<class_2960> extraModelConsumer) {
		pluginContext.extraModels.forEach(extraModelConsumer);
	}

	@Nullable
	public class_1100 modifyModelOnLoad(@Nullable class_1100 model, class_2960 id) {
		onLoadModifierContext.prepare(id);
		return pluginContext.modifyModelOnLoad().invoker().modifyModelOnLoad(model, onLoadModifierContext);
	}

	public class_1100 modifyModelBeforeBake(class_1100 model, class_2960 id, class_3665 settings, class_7775 baker) {
		if (beforeBakeModifierContextStack.isEmpty()) {
			beforeBakeModifierContextStack.add(new BeforeBakeModifierContext());
		}

		BeforeBakeModifierContext context = beforeBakeModifierContextStack.pop();
		context.prepare(id, settings, baker);

		model = pluginContext.modifyModelBeforeBake().invoker().modifyModelBeforeBake(model, context);

		beforeBakeModifierContextStack.push(context);
		return model;
	}

	public class_1087 modifyModelAfterBake(class_1087 model, class_2960 id, class_1100 sourceModel, class_3665 settings, class_7775 baker) {
		if (afterBakeModifierContextStack.isEmpty()) {
			afterBakeModifierContextStack.add(new AfterBakeModifierContext());
		}

		AfterBakeModifierContext context = afterBakeModifierContextStack.pop();
		context.prepare(id, sourceModel, settings, baker);

		model = pluginContext.modifyModelAfterBake().invoker().modifyModelAfterBake(model, context);

		afterBakeModifierContextStack.push(context);
		return model;
	}

	public class_9824.class_10095 modifyBlockModelsOnLoad(class_9824.class_10095 models) {
		Map<class_1091, class_9824.class_9825> map = models.comp_3063();

		if (!(map instanceof HashMap)) {
			map = new HashMap<>(map);
			models = new class_9824.class_10095(map);
		}

		putResolvedBlockStates(map);

		map.replaceAll((id, blockModel) -> {
			class_9979 original = blockModel.comp_2871();
			class_9979 modified = modifyBlockModelOnLoad(original, id, blockModel.comp_3062());

			if (original != modified) {
				return new class_9824.class_9825(blockModel.comp_3062(), modified);
			}

			return blockModel;
		});

		return models;
	}

	private void putResolvedBlockStates(Map<class_1091, class_9824.class_9825> map) {
		pluginContext.blockStateResolvers.forEach((block, resolver) -> {
			Optional<class_5321<class_2248>> optionalKey = class_7923.field_41175.method_29113(block);

			if (optionalKey.isEmpty()) {
				return;
			}

			class_2960 blockId = optionalKey.get().method_29177();

			resolveBlockStates(resolver, block, (state, model) -> {
				class_1091 modelId = class_773.method_3336(blockId, state);
				map.put(modelId, new class_9824.class_9825(state, model));
			});
		});
	}

	private void resolveBlockStates(BlockStateResolver resolver, class_2248 block, BiConsumer<class_2680, class_9979> output) {
		BlockStateResolverContext context = blockStateResolverContext;
		context.prepare(block);

		Reference2ReferenceMap<class_2680, class_9979> resolvedModels = context.models;
		ImmutableList<class_2680> allStates = block.method_9595().method_11662();
		boolean thrown = false;

		try {
			resolver.resolveBlockStates(context);
		} catch (Exception e) {
			LOGGER.error("Failed to resolve block state models for block {}. Using missing model for all states.", block, e);
			thrown = true;
		}

		if (!thrown) {
			if (resolvedModels.size() == allStates.size()) {
				// If there are as many resolved models as total states, all states have
				// been resolved and models do not need to be null-checked.
				resolvedModels.forEach(output);
			} else {
				for (class_2680 state : allStates) {
					@Nullable
					class_9979 model = resolvedModels.get(state);

					if (model == null) {
						LOGGER.error("Block state resolver did not provide a model for state {} in block {}. Using missing model.", state, block);
					} else {
						output.accept(state, model);
					}
				}
			}
		}

		resolvedModels.clear();
	}

	private class_9979 modifyBlockModelOnLoad(class_9979 model, class_1091 id, class_2680 state) {
		onLoadBlockModifierContext.prepare(id, state);
		return pluginContext.modifyBlockModelOnLoad().invoker().modifyModelOnLoad(model, onLoadBlockModifierContext);
	}

	public class_9979 modifyBlockModelBeforeBake(class_9979 model, class_1091 id, class_7775 baker) {
		beforeBakeBlockModifierContext.prepare(id, baker);
		return pluginContext.modifyBlockModelBeforeBake().invoker().modifyModelBeforeBake(model, beforeBakeBlockModifierContext);
	}

	public class_1087 modifyBlockModelAfterBake(class_1087 model, class_1091 id, class_9979 sourceModel, class_7775 baker) {
		afterBakeBlockModifierContext.prepare(id, sourceModel, baker);
		return pluginContext.modifyBlockModelAfterBake().invoker().modifyModelAfterBake(model, afterBakeBlockModifierContext);
	}

	private static class BlockStateResolverContext implements BlockStateResolver.Context {
		private class_2248 block;
		private final Reference2ReferenceMap<class_2680, class_9979> models = new Reference2ReferenceOpenHashMap<>();

		private void prepare(class_2248 block) {
			this.block = block;
			models.clear();
		}

		@Override
		public class_2248 block() {
			return block;
		}

		@Override
		public void setModel(class_2680 state, class_9979 model) {
			Objects.requireNonNull(state, "state cannot be null");
			Objects.requireNonNull(model, "model cannot be null");

			if (!state.method_27852(block)) {
				throw new IllegalArgumentException("Attempted to set model for state " + state + " on block " + block);
			}

			if (models.putIfAbsent(state, model) != null) {
				throw new IllegalStateException("Duplicate model for state " + state + " on block " + block);
			}
		}
	}

	private static class OnLoadModifierContext implements ModelModifier.OnLoad.Context {
		private class_2960 id;

		private void prepare(class_2960 id) {
			this.id = id;
		}

		@Override
		public class_2960 id() {
			return id;
		}
	}

	private static class BeforeBakeModifierContext implements ModelModifier.BeforeBake.Context {
		private class_2960 id;
		private class_3665 settings;
		private class_7775 baker;

		private void prepare(class_2960 id, class_3665 settings, class_7775 baker) {
			this.id = id;
			this.settings = settings;
			this.baker = baker;
		}

		@Override
		public class_2960 id() {
			return id;
		}

		@Override
		public class_3665 settings() {
			return settings;
		}

		@Override
		public class_7775 baker() {
			return baker;
		}
	}

	private static class AfterBakeModifierContext implements ModelModifier.AfterBake.Context {
		private class_2960 id;
		private class_1100 sourceModel;
		private class_3665 settings;
		private class_7775 baker;

		private void prepare(class_2960 id, class_1100 sourceModel, class_3665 settings, class_7775 baker) {
			this.id = id;
			this.sourceModel = sourceModel;
			this.settings = settings;
			this.baker = baker;
		}

		@Override
		public class_2960 id() {
			return id;
		}

		@Override
		public class_1100 sourceModel() {
			return sourceModel;
		}

		@Override
		public class_3665 settings() {
			return settings;
		}

		@Override
		public class_7775 baker() {
			return baker;
		}
	}

	private static class OnLoadBlockModifierContext implements ModelModifier.OnLoadBlock.Context {
		private class_1091 id;
		private class_2680 state;

		private void prepare(class_1091 id, class_2680 state) {
			this.id = id;
			this.state = state;
		}

		@Override
		public class_1091 id() {
			return id;
		}

		@Override
		public class_2680 state() {
			return state;
		}
	}

	private static class BeforeBakeBlockModifierContext implements ModelModifier.BeforeBakeBlock.Context {
		private class_1091 id;
		private class_7775 baker;

		private void prepare(class_1091 id, class_7775 baker) {
			this.id = id;
			this.baker = baker;
		}

		@Override
		public class_1091 id() {
			return id;
		}

		@Override
		public class_7775 baker() {
			return baker;
		}
	}

	private static class AfterBakeBlockModifierContext implements ModelModifier.AfterBakeBlock.Context {
		private class_1091 id;
		private class_9979 sourceModel;
		private class_7775 baker;

		private void prepare(class_1091 id, class_9979 sourceModel, class_7775 baker) {
			this.id = id;
			this.sourceModel = sourceModel;
			this.baker = baker;
		}

		@Override
		public class_1091 id() {
			return id;
		}

		@Override
		public class_9979 sourceModel() {
			return sourceModel;
		}

		@Override
		public class_7775 baker() {
			return baker;
		}
	}
}
