/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.model.loading;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.BiConsumer;

import com.google.common.collect.ImmutableList;
import com.llamalad7.mixinextras.injector.wrapoperation.Operation;
import it.unimi.dsi.fastutil.objects.Reference2ReferenceMap;
import it.unimi.dsi.fastutil.objects.Reference2ReferenceOpenHashMap;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.client.model.loading.v1.BlockStateResolver;
import net.fabricmc.fabric.api.client.model.loading.v1.ModelLoadingPlugin;
import net.fabricmc.fabric.api.client.model.loading.v1.ModelModifier;
import net.minecraft.class_10439;
import net.minecraft.class_1087;
import net.minecraft.class_1100;
import net.minecraft.class_2248;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_7775;
import net.minecraft.class_9824;

public class ModelLoadingEventDispatcher {
	private static final Logger LOGGER = LoggerFactory.getLogger(ModelLoadingEventDispatcher.class);
	public static final ThreadLocal<ModelLoadingEventDispatcher> CURRENT = new ThreadLocal<>();

	private final ModelLoadingPluginContextImpl pluginContext;

	private final BlockStateResolverContext blockStateResolverContext = new BlockStateResolverContext();
	private final OnLoadModifierContext onLoadModifierContext = new OnLoadModifierContext();
	private final OnLoadBlockModifierContext onLoadBlockModifierContext = new OnLoadBlockModifierContext();

	public ModelLoadingEventDispatcher(List<ModelLoadingPlugin> plugins) {
		this.pluginContext = new ModelLoadingPluginContextImpl();

		for (ModelLoadingPlugin plugin : plugins) {
			try {
				plugin.initialize(pluginContext);
			} catch (Exception exception) {
				LOGGER.error("Failed to initialize model loading plugin", exception);
			}
		}
	}

	public Map<class_2960, class_1100> modifyModelsOnLoad(Map<class_2960, class_1100> models) {
		if (!(models instanceof HashMap)) {
			models = new HashMap<>(models);
		}

		models.replaceAll(this::modifyModelOnLoad);
		return models;
	}

	private class_1100 modifyModelOnLoad(class_2960 id, class_1100 model) {
		onLoadModifierContext.prepare(id);
		return pluginContext.modifyModelOnLoad().invoker().modifyModelOnLoad(model, onLoadModifierContext);
	}

	public class_9824.class_10095 modifyBlockModelsOnLoad(class_9824.class_10095 models) {
		Map<class_2680, class_1087.class_9979> map = models.comp_3063();

		if (!(map instanceof HashMap)) {
			map = new HashMap<>(map);
			models = new class_9824.class_10095(map);
		}

		putResolvedBlockStates(map);
		map.replaceAll(this::modifyBlockModelOnLoad);

		return models;
	}

	private void putResolvedBlockStates(Map<class_2680, class_1087.class_9979> map) {
		pluginContext.blockStateResolvers.forEach((block, resolver) -> {
			resolveBlockStates(resolver, block, map::put);
		});
	}

	private void resolveBlockStates(BlockStateResolver resolver, class_2248 block, BiConsumer<class_2680, class_1087.class_9979> output) {
		BlockStateResolverContext context = blockStateResolverContext;
		context.prepare(block);

		Reference2ReferenceMap<class_2680, class_1087.class_9979> resolvedModels = context.models;
		ImmutableList<class_2680> allStates = block.method_9595().method_11662();
		boolean thrown = false;

		try {
			resolver.resolveBlockStates(context);
		} catch (Exception e) {
			LOGGER.error("Failed to resolve block state models for block {}. Using missing model for all states.", block, e);
			thrown = true;
		}

		if (!thrown) {
			if (resolvedModels.size() == allStates.size()) {
				// If there are as many resolved models as total states, all states have
				// been resolved and models do not need to be null-checked.
				resolvedModels.forEach(output);
			} else {
				for (class_2680 state : allStates) {
					@Nullable
					class_1087.class_9979 model = resolvedModels.get(state);

					if (model == null) {
						LOGGER.error("Block state resolver did not provide a model for state {} in block {}. Using missing model.", state, block);
					} else {
						output.accept(state, model);
					}
				}
			}
		}

		resolvedModels.clear();
	}

	private class_1087.class_9979 modifyBlockModelOnLoad(class_2680 state, class_1087.class_9979 model) {
		onLoadBlockModifierContext.prepare(state);
		return pluginContext.modifyBlockModelOnLoad().invoker().modifyModelOnLoad(model, onLoadBlockModifierContext);
	}

	public class_1087 modifyBlockModel(class_1087.class_9979 unbakedModel, class_2680 state, class_7775 baker, Operation<class_1087> bakeOperation) {
		BakeBlockModifierContext modifierContext = new BakeBlockModifierContext(state, baker);
		unbakedModel = pluginContext.modifyBlockModelBeforeBake().invoker().modifyModelBeforeBake(unbakedModel, modifierContext);
		class_1087 model = bakeOperation.call(unbakedModel, state, baker);
		modifierContext.prepareAfterBake(unbakedModel);
		return pluginContext.modifyBlockModelAfterBake().invoker().modifyModelAfterBake(model, modifierContext);
	}

	public class_10439 modifyItemModel(class_10439.class_10441 unbakedModel, class_2960 itemId, class_10439.class_10440 bakeContext, Operation<class_10439> bakeOperation) {
		BakeItemModifierContext modifierContext = new BakeItemModifierContext(itemId, bakeContext);
		unbakedModel = pluginContext.modifyItemModelBeforeBake().invoker().modifyModelBeforeBake(unbakedModel, modifierContext);
		class_10439 model = bakeOperation.call(unbakedModel, bakeContext);
		modifierContext.prepareAfterBake(unbakedModel);
		return pluginContext.modifyItemModelAfterBake().invoker().modifyModelAfterBake(model, modifierContext);
	}

	private static class BlockStateResolverContext implements BlockStateResolver.Context {
		private class_2248 block;
		private final Reference2ReferenceMap<class_2680, class_1087.class_9979> models = new Reference2ReferenceOpenHashMap<>();

		private void prepare(class_2248 block) {
			this.block = block;
			models.clear();
		}

		@Override
		public class_2248 block() {
			return block;
		}

		@Override
		public void setModel(class_2680 state, class_1087.class_9979 model) {
			Objects.requireNonNull(state, "state cannot be null");
			Objects.requireNonNull(model, "model cannot be null");

			if (!state.method_27852(block)) {
				throw new IllegalArgumentException("Attempted to set model for state " + state + " on block " + block);
			}

			if (models.putIfAbsent(state, model) != null) {
				throw new IllegalStateException("Duplicate model for state " + state + " on block " + block);
			}
		}
	}

	private static class OnLoadModifierContext implements ModelModifier.OnLoad.Context {
		private class_2960 id;

		private void prepare(class_2960 id) {
			this.id = id;
		}

		@Override
		public class_2960 id() {
			return id;
		}
	}

	private static class OnLoadBlockModifierContext implements ModelModifier.OnLoadBlock.Context {
		private class_2680 state;

		private void prepare(class_2680 state) {
			this.state = state;
		}

		@Override
		public class_2680 state() {
			return state;
		}
	}

	private static class BakeBlockModifierContext implements ModelModifier.BeforeBakeBlock.Context, ModelModifier.AfterBakeBlock.Context {
		private final class_2680 state;
		private final class_7775 baker;
		private class_1087.class_9979 sourceModel;

		private BakeBlockModifierContext(class_2680 state, class_7775 baker) {
			this.state = state;
			this.baker = baker;
		}

		private void prepareAfterBake(class_1087.class_9979 sourceModel) {
			this.sourceModel = sourceModel;
		}

		@Override
		public class_2680 state() {
			return state;
		}

		@Override
		public class_7775 baker() {
			return baker;
		}

		@Override
		public class_1087.class_9979 sourceModel() {
			return sourceModel;
		}
	}

	private static class BakeItemModifierContext implements ModelModifier.BeforeBakeItem.Context, ModelModifier.AfterBakeItem.Context {
		private final class_2960 itemId;
		private final class_10439.class_10440 bakeContext;
		private class_10439.class_10441 sourceModel;

		private BakeItemModifierContext(class_2960 itemId, class_10439.class_10440 bakeContext) {
			this.itemId = itemId;
			this.bakeContext = bakeContext;
		}

		private void prepareAfterBake(class_10439.class_10441 sourceModel) {
			this.sourceModel = sourceModel;
		}

		@Override
		public class_2960 itemId() {
			return itemId;
		}

		@Override
		public class_10439.class_10440 bakeContext() {
			return bakeContext;
		}

		@Override
		public class_10439.class_10441 sourceModel() {
			return sourceModel;
		}
	}
}
