/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.networking.v1;

import java.util.Objects;

import io.netty.channel.ChannelFutureListener;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.impl.networking.GenericFutureListenerHolder;
import net.minecraft.class_2540;
import net.minecraft.class_2596;
import net.minecraft.class_2960;
import net.minecraft.class_7648;

/**
 * Represents something that supports sending packets to channels.
 * @see PacketByteBufs
 */
public interface PacketSender {
	/**
	 * Makes a packet for a channel.
	 *
	 * @param channelName the id of the channel
	 * @param buf     the content of the packet
	 */
	class_2596<?> createPacket(class_2960 channelName, class_2540 buf);

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 */
	void sendPacket(class_2596<?> packet);

	/**
	 * Sends a packet.
	 * @param packet the packet
	 */
	default <T extends FabricPacket> void sendPacket(T packet) {
		class_2540 buf = PacketByteBufs.create();
		packet.write(buf);
		sendPacket(packet.getType().getId(), buf);
	}

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}. The callback may also accept a {@link ChannelFutureListener}.
	 */
	void sendPacket(class_2596<?> packet, @Nullable GenericFutureListener<? extends Future<? super Void>> callback);

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}. The callback may also accept a {@link ChannelFutureListener}.
	 */
	default <T extends FabricPacket> void sendPacket(T packet, @Nullable GenericFutureListener<? extends Future<? super Void>> callback) {
		class_2540 buf = PacketByteBufs.create();
		packet.write(buf);
		sendPacket(packet.getType().getId(), buf, callback);
	}

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}. The callback may also accept a {@link ChannelFutureListener}.
	 */
	void sendPacket(class_2596<?> packet, @Nullable class_7648 callback);

	/**
	 * Sends a packet.
	 *
	 * @param packet the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}. The callback may also accept a {@link ChannelFutureListener}.
	 */
	default <T extends FabricPacket> void sendPacket(T packet, @Nullable class_7648 callback) {
		class_2540 buf = PacketByteBufs.create();
		packet.write(buf);
		sendPacket(packet.getType().getId(), buf, callback);
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel the id of the channel
	 * @param buf the content of the packet
	 */
	default void sendPacket(class_2960 channel, class_2540 buf) {
		Objects.requireNonNull(channel, "Channel cannot be null");
		Objects.requireNonNull(buf, "Payload cannot be null");

		this.sendPacket(this.createPacket(channel, buf));
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel  the id of the channel
	 * @param buf the content of the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}
	 */
	// the generic future listener can accept ChannelFutureListener
	default void sendPacket(class_2960 channel, class_2540 buf, @Nullable GenericFutureListener<? extends Future<? super Void>> callback) {
		sendPacket(channel, buf, GenericFutureListenerHolder.create(callback));
	}

	/**
	 * Sends a packet to a channel.
	 *
	 * @param channel  the id of the channel
	 * @param buf the content of the packet
	 * @param callback an optional callback to execute after the packet is sent, may be {@code null}
	 */
	default void sendPacket(class_2960 channel, class_2540 buf, @Nullable class_7648 callback) {
		Objects.requireNonNull(channel, "Channel cannot be null");
		Objects.requireNonNull(buf, "Payload cannot be null");

		this.sendPacket(this.createPacket(channel, buf), callback);
	}
}
