/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.networking;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.netty.util.AsciiString;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.networking.v1.PacketByteBufs;
import net.fabricmc.fabric.api.networking.v1.PacketSender;
import net.minecraft.class_151;
import net.minecraft.class_2535;
import net.minecraft.class_2540;
import net.minecraft.class_2596;
import net.minecraft.class_2960;
import net.minecraft.class_7648;

/**
 * A network addon which is aware of the channels the other side may receive.
 *
 * @param <H> the channel handler type
 */
public abstract class AbstractChanneledNetworkAddon<H> extends AbstractNetworkAddon<H> implements PacketSender {
	protected final class_2535 connection;
	protected final GlobalReceiverRegistry<H> receiver;
	protected final Set<class_2960> sendableChannels;
	protected final Set<class_2960> sendableChannelsView;

	protected AbstractChanneledNetworkAddon(GlobalReceiverRegistry<H> receiver, class_2535 connection, String description) {
		this(receiver, connection, new HashSet<>(), description);
	}

	protected AbstractChanneledNetworkAddon(GlobalReceiverRegistry<H> receiver, class_2535 connection, Set<class_2960> sendableChannels, String description) {
		super(receiver, description);
		this.connection = connection;
		this.receiver = receiver;
		this.sendableChannels = sendableChannels;
		this.sendableChannelsView = Collections.unmodifiableSet(sendableChannels);
	}

	public abstract void lateInit();

	protected void registerPendingChannels(ChannelInfoHolder holder) {
		final Collection<class_2960> pending = holder.getPendingChannelsNames();

		if (!pending.isEmpty()) {
			register(new ArrayList<>(pending));
			pending.clear();
		}
	}

	// always supposed to handle async!
	protected boolean handle(class_2960 channelName, class_2540 originalBuf) {
		this.logger.debug("Handling inbound packet from channel with name \"{}\"", channelName);

		// Handle reserved packets
		if (NetworkingImpl.REGISTER_CHANNEL.equals(channelName)) {
			this.receiveRegistration(true, PacketByteBufs.slice(originalBuf));
			return true;
		}

		if (NetworkingImpl.UNREGISTER_CHANNEL.equals(channelName)) {
			this.receiveRegistration(false, PacketByteBufs.slice(originalBuf));
			return true;
		}

		@Nullable H handler = this.getHandler(channelName);

		if (handler == null) {
			return false;
		}

		class_2540 buf = PacketByteBufs.slice(originalBuf);

		try {
			this.receive(handler, buf);
		} catch (Throwable ex) {
			this.logger.error("Encountered exception while handling in channel with name \"{}\"", channelName, ex);
			throw ex;
		}

		return true;
	}

	protected abstract void receive(H handler, class_2540 buf);

	protected void sendInitialChannelRegistrationPacket() {
		final class_2540 buf = this.createRegistrationPacket(this.getReceivableChannels());

		if (buf != null) {
			this.sendPacket(NetworkingImpl.REGISTER_CHANNEL, buf);
		}
	}

	@Nullable
	protected class_2540 createRegistrationPacket(Collection<class_2960> channels) {
		if (channels.isEmpty()) {
			return null;
		}

		class_2540 buf = PacketByteBufs.create();
		boolean first = true;

		for (class_2960 channel : channels) {
			if (first) {
				first = false;
			} else {
				buf.writeByte(0);
			}

			buf.writeBytes(channel.toString().getBytes(StandardCharsets.US_ASCII));
		}

		return buf;
	}

	// wrap in try with res (buf)
	protected void receiveRegistration(boolean register, class_2540 buf) {
		List<class_2960> ids = new ArrayList<>();
		StringBuilder active = new StringBuilder();

		while (buf.isReadable()) {
			byte b = buf.readByte();

			if (b != 0) {
				active.append(AsciiString.b2c(b));
			} else {
				this.addId(ids, active);
				active = new StringBuilder();
			}
		}

		this.addId(ids, active);
		this.schedule(register ? () -> register(ids) : () -> unregister(ids));
	}

	void register(List<class_2960> ids) {
		this.sendableChannels.addAll(ids);
		this.invokeRegisterEvent(ids);
	}

	void unregister(List<class_2960> ids) {
		this.sendableChannels.removeAll(ids);
		this.invokeUnregisterEvent(ids);
	}

	@Override
	public void sendPacket(class_2596<?> packet) {
		Objects.requireNonNull(packet, "Packet cannot be null");

		this.connection.method_10743(packet);
	}

	@Override
	public void sendPacket(class_2596<?> packet, @Nullable GenericFutureListener<? extends Future<? super Void>> callback) {
		sendPacket(packet, GenericFutureListenerHolder.create(callback));
	}

	@Override
	public void sendPacket(class_2596<?> packet, class_7648 callback) {
		Objects.requireNonNull(packet, "Packet cannot be null");

		this.connection.method_10752(packet, callback);
	}

	/**
	 * Schedules a task to run on the main thread.
	 */
	protected abstract void schedule(Runnable task);

	protected abstract void invokeRegisterEvent(List<class_2960> ids);

	protected abstract void invokeUnregisterEvent(List<class_2960> ids);

	private void addId(List<class_2960> ids, StringBuilder sb) {
		String literal = sb.toString();

		try {
			ids.add(new class_2960(literal));
		} catch (class_151 ex) {
			this.logger.warn("Received invalid channel identifier \"{}\" from connection {}", literal, this.connection);
		}
	}

	public Set<class_2960> getSendableChannels() {
		return this.sendableChannelsView;
	}
}
