/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.networking;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import io.netty.buffer.ByteBufUtil;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.networking.v1.PayloadTypeRegistry;
import net.fabricmc.fabric.impl.networking.splitter.FabricPacketSplitter;
import net.minecraft.class_2539;
import net.minecraft.class_2540;
import net.minecraft.class_2598;
import net.minecraft.class_2960;
import net.minecraft.class_8703;
import net.minecraft.class_8710;
import net.minecraft.class_9127;
import net.minecraft.class_9129;
import net.minecraft.class_9139;

public class PayloadTypeRegistryImpl<B extends class_2540> implements PayloadTypeRegistry<B> {
	public static final PayloadTypeRegistryImpl<class_2540> CONFIGURATION_C2S = new PayloadTypeRegistryImpl<>(class_2539.field_45671, class_2598.field_11941);
	public static final PayloadTypeRegistryImpl<class_2540> CONFIGURATION_S2C = new PayloadTypeRegistryImpl<>(class_2539.field_45671, class_2598.field_11942);
	public static final PayloadTypeRegistryImpl<class_9129> PLAY_C2S = new PayloadTypeRegistryImpl<>(class_2539.field_20591, class_2598.field_11941);
	public static final PayloadTypeRegistryImpl<class_9129> PLAY_S2C = new PayloadTypeRegistryImpl<>(class_2539.field_20591, class_2598.field_11942);
	private final Map<class_2960, class_8710.class_9155<B, ? extends class_8710>> packetTypes = new HashMap<>();
	private final Object2IntMap<class_2960> maxPacketSize = new Object2IntOpenHashMap<>();
	private final class_2539 state;
	private final class_2598 side;
	private final int minimalSplittableSize;

	private PayloadTypeRegistryImpl(class_2539 state, class_2598 side) {
		this.state = state;
		this.side = side;
		this.minimalSplittableSize = side == class_2598.field_11942 ? FabricPacketSplitter.SAFE_S2C_SPLIT_SIZE : FabricPacketSplitter.SAFE_C2S_SPLIT_SIZE;
	}

	@Nullable
	public static PayloadTypeRegistryImpl<?> get(class_9127<?> state) {
		return switch (state.comp_2234()) {
		case field_45671 -> state.comp_2235() == class_2598.field_11942 ? CONFIGURATION_S2C : CONFIGURATION_C2S;
		case field_20591 -> state.comp_2235() == class_2598.field_11942 ? PLAY_S2C : PLAY_C2S;
		default -> null;
		};
	}

	@Override
	public <T extends class_8710> class_8710.class_9155<? super B, T> register(class_8710.class_9154<T> id, class_9139<? super B, T> codec) {
		Objects.requireNonNull(id, "id");
		Objects.requireNonNull(codec, "codec");

		final class_8710.class_9155<B, T> payloadType = new class_8710.class_9155<>(id, codec.method_56430());

		if (packetTypes.containsKey(id.comp_2242())) {
			throw new IllegalArgumentException("Packet type " + id + " is already registered!");
		}

		packetTypes.put(id.comp_2242(), payloadType);
		return payloadType;
	}

	@Override
	public <T extends class_8710> class_8710.class_9155<? super B, T> registerLarge(class_8710.class_9154<T> id, class_9139<? super B, T> codec, int maxPayloadSize) {
		if (maxPayloadSize < 0) {
			throw new IllegalArgumentException("Provided maxPayloadSize needs to be positive!");
		}

		class_8710.class_9155<? super B, T> type = register(id, codec);
		// Defines max packet size, increased by length of packet's Identifier to cover full size of CustomPayloadX2YPackets.
		int identifierSize = ByteBufUtil.utf8MaxBytes(id.comp_2242().toString());
		int maxPacketSize = maxPayloadSize + class_8703.method_53015(identifierSize) + identifierSize + 5 * 2;

		// Prevent overflow
		if (maxPacketSize < 0) {
			maxPacketSize = Integer.MAX_VALUE;
		}

		// No need to enable splitting, if packet's max size is smaller than chunk
		if (maxPacketSize > this.minimalSplittableSize) {
			this.maxPacketSize.put(id.comp_2242(), maxPacketSize);
		}

		return type;
	}

	@Nullable
	public class_8710.class_9155<B, ? extends class_8710> get(class_2960 id) {
		return packetTypes.get(id);
	}

	@Nullable
	public <T extends class_8710> class_8710.class_9155<B, T> get(class_8710.class_9154<T> id) {
		//noinspection unchecked
		return (class_8710.class_9155<B, T>) packetTypes.get(id.comp_2242());
	}

	public int getMaxPacketSize(class_2960 id) {
		return this.maxPacketSize.getOrDefault(id, -1);
	}

	public class_2539 getPhase() {
		return state;
	}

	public class_2598 getSide() {
		return side;
	}
}
