/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.networking;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import io.netty.buffer.Unpooled;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.api.network.PacketConsumer;
import net.fabricmc.fabric.api.network.PacketContext;
import net.fabricmc.fabric.api.network.PacketRegistry;
import net.minecraft.class_151;
import net.minecraft.class_2540;
import net.minecraft.class_2596;
import net.minecraft.class_2960;

public abstract class PacketRegistryImpl implements PacketRegistry {
	protected static final Logger LOGGER = LogManager.getLogger();
	protected final Map<class_2960, PacketConsumer> consumerMap;

	PacketRegistryImpl() {
		consumerMap = new LinkedHashMap<>();
	}

	public static Optional<class_2596<?>> createInitialRegisterPacket(PacketRegistry registry) {
		PacketRegistryImpl impl = (PacketRegistryImpl) registry;
		return impl.createRegisterTypePacket(PacketTypes.REGISTER, impl.consumerMap.keySet());
	}

	@Override
	public void register(class_2960 id, PacketConsumer consumer) {
		boolean isNew = true;

		if (consumerMap.containsKey(id)) {
			LOGGER.warn("Registered duplicate packet " + id + "!");
			LOGGER.trace(new Throwable());
			isNew = false;
		}

		consumerMap.put(id, consumer);

		if (isNew) {
			onRegister(id);
		}
	}

	@Override
	public void unregister(class_2960 id) {
		if (consumerMap.remove(id) != null) {
			onUnregister(id);
		} else {
			LOGGER.warn("Tried to unregister non-registered packet " + id + "!");
			LOGGER.trace(new Throwable());
		}
	}

	protected abstract void onRegister(class_2960 id);

	protected abstract void onUnregister(class_2960 id);

	protected abstract Collection<class_2960> getIdCollectionFor(PacketContext context);

	protected abstract void onReceivedRegisterPacket(PacketContext context, Collection<class_2960> ids);

	protected abstract void onReceivedUnregisterPacket(PacketContext context, Collection<class_2960> ids);

	protected Optional<class_2596<?>> createRegisterTypePacket(class_2960 id, Collection<class_2960> ids) {
		if (ids.isEmpty()) {
			return Optional.empty();
		}

		class_2540 buf = new class_2540(Unpooled.buffer());
		boolean first = true;

		for (class_2960 a : ids) {
			if (!first) {
				buf.writeByte(0);
			} else {
				first = false;
			}

			buf.writeBytes(a.toString().getBytes(StandardCharsets.US_ASCII));
		}

		return Optional.of(toPacket(id, buf));
	}

	private boolean acceptRegisterType(class_2960 id, PacketContext context, Supplier<class_2540> bufSupplier) {
		Collection<class_2960> ids = new HashSet<>();

		{
			StringBuilder sb = new StringBuilder();
			char c;
			class_2540 buf = bufSupplier.get();

			try {
				while (buf.readerIndex() < buf.writerIndex()) {
					c = (char) buf.readByte();

					if (c == 0) {
						String s = sb.toString();

						if (!s.isEmpty()) {
							try {
								ids.add(new class_2960(s));
							} catch (class_151 e) {
								LOGGER.warn("Received invalid identifier in " + id + ": " + s + " (" + e.getLocalizedMessage() + ")");
								LOGGER.trace(e);
							}
						}

						sb = new StringBuilder();
					} else {
						sb.append(c);
					}
				}
			} finally {
				buf.release();
			}

			String s = sb.toString();

			if (!s.isEmpty()) {
				try {
					ids.add(new class_2960(s));
				} catch (class_151 e) {
					LOGGER.warn("Received invalid identifier in " + id + ": " + s + " (" + e.getLocalizedMessage() + ")");
					LOGGER.trace(e);
				}
			}
		}

		Collection<class_2960> target = getIdCollectionFor(context);

		if (id.equals(PacketTypes.UNREGISTER)) {
			target.removeAll(ids);
			onReceivedUnregisterPacket(context, ids);
		} else {
			target.addAll(ids);
			onReceivedRegisterPacket(context, ids);
		}

		return false; // continue execution for other mods
	}

	/**
	 * Hook for accepting packets used in Fabric mixins.
	 *
	 * <p>As PacketByteBuf getters in vanilla create a copy (to allow releasing the original packet buffer without
	 * breaking other, potentially delayed accesses), we use a Supplier to generate those copies and release them
	 * when needed.
	 *
	 * @param id      The packet Identifier received.
	 * @param context The packet context provided.
	 * @param bufSupplier A supplier creating a new PacketByteBuf.
	 * @return Whether or not the packet was handled by this packet registry.
	 */
	public boolean accept(class_2960 id, PacketContext context, Supplier<class_2540> bufSupplier) {
		if (id.equals(PacketTypes.REGISTER) || id.equals(PacketTypes.UNREGISTER)) {
			return acceptRegisterType(id, context, bufSupplier);
		}

		PacketConsumer consumer = consumerMap.get(id);

		if (consumer != null) {
			class_2540 buf = bufSupplier.get();

			try {
				consumer.accept(context, buf);
			} catch (Throwable t) {
				LOGGER.warn("Failed to handle packet " + id + "!", t);
			} finally {
				if (buf.refCnt() > 0 && !PacketDebugOptions.DISABLE_BUFFER_RELEASES) {
					buf.release();
				}
			}

			return true;
		} else {
			return false;
		}
	}
}
