/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.server;

import java.util.stream.Stream;
import net.minecraft.class_1297;
import net.minecraft.class_1657;
import net.minecraft.class_1923;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2586;
import net.minecraft.class_2802;
import net.minecraft.class_3215;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3898;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.impl.networking.server.EntityTrackerStorageAccessor;

/**
 * Helper streams for looking up players on a server.
 *
 * <p>In general, most of these methods will only function with a {@link ServerWorld} instance.
 */
public final class PlayerStream {
	private PlayerStream() { }

	public static Stream<class_3222> all(MinecraftServer server) {
		if (server.method_3760() != null) {
			return server.method_3760().method_14571().stream();
		} else {
			return Stream.empty();
		}
	}

	public static Stream<class_1657> world(class_1937 world) {
		if (world instanceof class_3218) {
			// noinspection unchecked
			return ((Stream) ((class_3218) world).method_18456().stream());
		} else {
			throw new RuntimeException("Only supported on ServerWorld!");
		}
	}

	public static Stream<class_1657> watching(class_1937 world, class_1923 pos) {
		class_2802 manager = world.method_8398();

		if (!(manager instanceof class_3215)) {
			throw new RuntimeException("Only supported on ServerWorld!");
		} else {
			//noinspection unchecked
			return ((Stream) ((class_3215) manager).field_17254.method_17210(pos, false));
		}
	}

	/**
	 * Warning: If the provided entity is a PlayerEntity themselves, it is not
	 * guaranteed by the contract that said PlayerEntity is included in the
	 * resulting stream.
	 */
	@SuppressWarnings("JavaDoc")
	public static Stream<class_1657> watching(class_1297 entity) {
		class_2802 manager = entity.method_5770().method_8398();

		if (manager instanceof class_3215) {
			class_3898 storage = ((class_3215) manager).field_17254;

			if (storage instanceof EntityTrackerStorageAccessor) {
				//noinspection unchecked
				return ((Stream) ((EntityTrackerStorageAccessor) storage).fabric_getTrackingPlayers(entity));
			}
		}

		// fallback
		return watching(entity.method_5770(), new class_1923((int) (entity.method_23317() / 16.0D), (int) (entity.method_23321() / 16.0D)));
	}

	public static Stream<class_1657> watching(class_2586 entity) {
		return watching(entity.method_10997(), entity.method_11016());
	}

	public static Stream<class_1657> watching(class_1937 world, class_2338 pos) {
		return watching(world, new class_1923(pos));
	}

	public static Stream<class_1657> around(class_1937 world, class_243 vector, double radius) {
		double radiusSq = radius * radius;
		return world(world).filter((p) -> p.method_5707(vector) <= radiusSq);
	}

	public static Stream<class_1657> around(class_1937 world, class_2338 pos, double radius) {
		double radiusSq = radius * radius;
		return world(world).filter((p) -> p.method_5649(pos.method_10263(), pos.method_10264(), pos.method_10260()) <= radiusSq);
	}
}
