/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.object.builder.v1.villager;

import java.util.Objects;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import net.fabricmc.fabric.mixin.object.builder.VillagerTypeAccessor;
import net.minecraft.class_1959;
import net.minecraft.class_2960;
import net.minecraft.class_3852;
import net.minecraft.class_3854;
import net.minecraft.class_5321;

/**
 * Utilities related to the creation of {@link class_3854}s.
 * Not to be confused with a {@link class_3852}, a villager type defines the appearance of a villager.
 *
 * <p>Creation and registration of custom villager types may be done by using {@link VillagerTypeHelper#register(class_2960)}.
 *
 * <p>Creation and registration of a villager type does not guarantee villagers of a specific type will be created in a world.
 * Typically the villager type is bound to a specific group of biomes.
 * To allow a villager type to be spawned in a specific biome, use {@link VillagerTypeHelper#addVillagerTypeToBiome(class_5321, class_3854)}.
 *
 * <p>The texture used for the appearance of the villager is located at {@code assets/IDENTIFIER_NAMESPACE/textures/entity/villager/type/IDENTIFIER_PATH.png}.
 */
public final class VillagerTypeHelper {
	private static final Logger LOGGER = LogManager.getLogger();

	/**
	 * Creates and registers a new villager type.
	 *
	 * @param id the id of the villager type
	 * @return a new villager type
	 */
	public static class_3854 register(class_2960 id) {
		Objects.requireNonNull(id, "Id of villager type cannot be null");
		return VillagerTypeAccessor.callRegister(id.toString());
	}

	/**
	 * Sets the biome a villager type can spawn in.
	 *
	 * @param biomeKey the registry key of the biome
	 * @param villagerType the villager type
	 */
	public static void addVillagerTypeToBiome(class_5321<class_1959> biomeKey, class_3854 villagerType) {
		Objects.requireNonNull(biomeKey, "Biome registry key cannot be null");
		Objects.requireNonNull(villagerType, "Villager type cannot be null");

		if (VillagerTypeAccessor.getBiomeTypeToIdMap().put(biomeKey, villagerType) != null) {
			LOGGER.debug("Overriding existing Biome -> VillagerType registration for Biome {}", biomeKey.method_29177().toString());
		}
	}

	private VillagerTypeHelper() {
	}
}
