/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.object.builder;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.event.registry.FabricRegistryBuilder;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.fabricmc.fabric.api.event.registry.RegistryIdRemapCallback;
import net.fabricmc.fabric.mixin.object.builder.TrackedDataHandlerRegistryAccessor;
import net.minecraft.class_2378;
import net.minecraft.class_2941;
import net.minecraft.class_2960;
import net.minecraft.class_3513;
import net.minecraft.class_5321;

public final class FabricTrackedDataRegistryImpl {
	private static final Logger LOGGER = LoggerFactory.getLogger(FabricTrackedDataRegistryImpl.class);

	private static final class_2960 HANDLER_REGISTRY_ID = class_2960.method_60655("fabric-object-builder-api-v1", "tracked_data_handler");
	private static final class_5321<class_2378<class_2941<?>>> HANDLER_REGISTRY_KEY = class_5321.method_29180(HANDLER_REGISTRY_ID);

	private static final List<class_2941<?>> VANILLA_HANDLERS = new ArrayList<>();
	@Nullable
	private static class_2378<class_2941<?>> handlerRegistry = null;
	private static final List<class_2941<?>> EXTERNAL_MODDED_HANDLERS = new ArrayList<>();

	private FabricTrackedDataRegistryImpl() {
	}

	public static boolean hasStoredVanillaHandlers() {
		return !VANILLA_HANDLERS.isEmpty();
	}

	public static void storeVanillaHandlers() {
		if (hasStoredVanillaHandlers()) {
			throw new IllegalStateException("Already stored vanilla handlers!");
		}

		class_3513<class_2941<?>> dataHandlers = TrackedDataHandlerRegistryAccessor.fabric_getDataHandlers();

		for (class_2941<?> handler : dataHandlers) {
			VANILLA_HANDLERS.add(handler);
		}

		LOGGER.debug("Stored {} vanilla handlers", VANILLA_HANDLERS.size());
	}

	private static void storeExternalHandlers() {
		class_3513<class_2941<?>> dataHandlers = TrackedDataHandlerRegistryAccessor.fabric_getDataHandlers();

		for (class_2941<?> handler : dataHandlers) {
			if (VANILLA_HANDLERS.contains(handler)) continue;
			if (handlerRegistry != null && handlerRegistry.method_10221(handler) != null) continue;
			if (EXTERNAL_MODDED_HANDLERS.contains(handler)) continue;

			EXTERNAL_MODDED_HANDLERS.add(handler);
			LOGGER.warn("Tracked data handler {} is not managed by vanilla or Fabric API; it may be prone to desynchronization!", handler);
		}
	}

	/**
	 * Reorders handlers in {@code TrackedDataHandlerRegistry#DATA_HANDLERS} to have a consistent order between client and server.
	 *
	 * <p>The order used is as follows:
	 *
	 * <ul>
	 *   <li>Vanilla handlers</li>
	 *   <li>Handlers in the Fabric API registry (sorted by ID)</li>
	 *   <li>External modded handlers</li>
	 * </ul>
	*/
	private static void reorderHandlers() {
		class_3513<class_2941<?>> dataHandlers = TrackedDataHandlerRegistryAccessor.fabric_getDataHandlers();
		LOGGER.debug("Reordering tracked data handlers containing {} entries", dataHandlers.method_10204());

		// Reset the map so that handlers can be added back in a new order
		dataHandlers.method_15229();

		// Add handlers back to map
		for (class_2941<?> handler : VANILLA_HANDLERS) {
			dataHandlers.method_15225(handler);
		}

		if (handlerRegistry != null) {
			for (class_2941<?> handler : handlerRegistry) {
				dataHandlers.method_15225(handler);
			}
		}

		for (class_2941<?> handler : EXTERNAL_MODDED_HANDLERS) {
			dataHandlers.method_15225(handler);
		}

		LOGGER.debug("Finished reordering tracked data handlers containing {} entries", dataHandlers.method_10204());
	}

	public static void register(class_2960 id, class_2941<?> handler) {
		Objects.requireNonNull(id, "Tracked data handler ID cannot be null!");
		Objects.requireNonNull(handler, "Tracked data handler cannot be null!");

		storeExternalHandlers();

		if (VANILLA_HANDLERS.contains(handler) || EXTERNAL_MODDED_HANDLERS.contains(handler)) {
			throw new IllegalArgumentException("Cannot register tracked data handler previously added via TrackedDataHandlerRegistry.register");
		}

		if (handlerRegistry == null) {
			handlerRegistry = FabricRegistryBuilder
					.createSimple(HANDLER_REGISTRY_KEY)
					.attribute(RegistryAttribute.SYNCED)
					.buildAndRegister();

			RegistryIdRemapCallback.event(handlerRegistry).register(state -> {
				storeExternalHandlers();
				reorderHandlers();
			});
		}

		class_2378.method_10230(handlerRegistry, id, handler);
		reorderHandlers();
	}

	@Nullable
	public static class_2941<?> get(class_2960 id) {
		Objects.requireNonNull(id, "Tracked data handler ID cannot be null!");

		if (handlerRegistry == null) {
			return null;
		}

		return handlerRegistry.method_63535(id);
	}

	@Nullable
	public static class_2960 getId(class_2941<?> handler) {
		Objects.requireNonNull(handler, "Tracked data handler cannot be null!");

		if (handlerRegistry == null) {
			return null;
		}

		return handlerRegistry.method_10221(handler);
	}
}
