/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.block;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import net.fabricmc.fabric.api.event.registry.BlockConstructedCallback;
import net.fabricmc.fabric.impl.tool.attribute.ToolManager;
import net.minecraft.class_1767;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2498;
import net.minecraft.class_2960;
import net.minecraft.class_3494;
import net.minecraft.class_3614;
import net.minecraft.class_3620;

/**
 * Fabric's version of Block.Settings. Adds additional methods and hooks
 * not found in the original class.
 *
 * <p>To use it, simply replace Block.Settings.create() with
 * FabricBlockSettings.create() and add .build() at the end to return the
 * vanilla Block.Settings instance beneath.
 */
public class FabricBlockSettings {
	static {
		BlockConstructedCallback.EVENT.register(FabricBlockSettings::onBuild);
	}

	private static final Map<class_2248.Settings, ExtraData> EXTRA_DATA = new HashMap<>();

	protected final class_2248.Settings delegate;

	static final class ExtraData {
		private final List<MiningLevel> miningLevels = new ArrayList<>();
		/* @Nullable */ private Boolean breakByHand;

		private ExtraData(class_2248.Settings settings) {
		}

		void breakByHand(boolean breakByHand) {
			this.breakByHand = breakByHand;
		}

		void addMiningLevel(class_3494<class_1792> tag, int level) {
			miningLevels.add(new MiningLevel(tag, level));
		}
	}

	private static final class MiningLevel {
		private final class_3494<class_1792> tag;
		private final int level;

		MiningLevel(class_3494<class_1792> tag, int level) {
			this.tag = tag;
			this.level = level;
		}
	}

	static ExtraData computeExtraData(class_2248.Settings settings) {
		return EXTRA_DATA.computeIfAbsent(settings, ExtraData::new);
	}

	private static void onBuild(class_2248.Settings settings, class_2248 block) {
		// TODO: Load only if fabric-mining-levels present
		ExtraData data = EXTRA_DATA.get(settings);

		if (data != null) {
			if (data.breakByHand != null) {
				ToolManager.entry(block).setBreakByHand(data.breakByHand);
			}

			for (MiningLevel tml : data.miningLevels) {
				ToolManager.entry(block).putBreakByTool(tml.tag, tml.level);
			}
		}
	}

	protected FabricBlockSettings(class_3614 material, class_3620 color) {
		this(class_2248.Settings.method_9639(material, color));
	}

	protected FabricBlockSettings(class_2248 base) {
		this(class_2248.Settings.method_9630(base));
	}

	protected FabricBlockSettings(final class_2248.Settings delegate) {
		this.delegate = delegate;
	}

	public static FabricBlockSettings of(class_3614 material) {
		return of(material, material.method_15803());
	}

	public static FabricBlockSettings of(class_3614 material, class_3620 color) {
		return new FabricBlockSettings(material, color);
	}

	public static FabricBlockSettings of(class_3614 material, class_1767 color) {
		return new FabricBlockSettings(material, color.method_7794());
	}

	public static FabricBlockSettings copy(class_2248 base) {
		return new FabricBlockSettings(base);
	}

	public static FabricBlockSettings copyOf(class_2248.Settings settings) {
		return new FabricBlockSettings(settings);
	}

	/* FABRIC HELPERS */

	public FabricBlockSettings breakByHand(boolean breakByHand) {
		computeExtraData(delegate).breakByHand(breakByHand);
		return this;
	}

	public FabricBlockSettings breakByTool(class_3494<class_1792> tag, int miningLevel) {
		computeExtraData(delegate).addMiningLevel(tag, miningLevel);
		return this;
	}

	public FabricBlockSettings breakByTool(class_3494<class_1792> tag) {
		return breakByTool(tag, 0);
	}

	/* DELEGATE WRAPPERS */

	public FabricBlockSettings materialColor(class_3620 color) {
		BlockSettingsExtensions.materialColor(delegate, color);
		return this;
	}

	public FabricBlockSettings materialColor(class_1767 color) {
		return materialColor(color.method_7794());
	}

	public FabricBlockSettings collidable(boolean collidable) {
		BlockSettingsExtensions.collidable(delegate, collidable);
		return this;
	}

	public FabricBlockSettings noCollision() {
		delegate.method_9634();
		return this;
	}

	public FabricBlockSettings nonOpaque() {
		delegate.method_22488();
		return this;
	}

	public FabricBlockSettings sounds(class_2498 group) {
		BlockSettingsExtensions.sounds(delegate, group);
		return this;
	}

	public FabricBlockSettings ticksRandomly() {
		BlockSettingsExtensions.ticksRandomly(delegate);
		return this;
	}

	public FabricBlockSettings lightLevel(int lightLevel) {
		BlockSettingsExtensions.lightLevel(delegate, lightLevel);
		return this;
	}

	public FabricBlockSettings hardness(float hardness) {
		BlockSettingsExtensions.hardness(delegate, hardness);
		return this;
	}

	public FabricBlockSettings resistance(float resistance) {
		BlockSettingsExtensions.resistance(delegate, resistance);
		return this;
	}

	public FabricBlockSettings strength(float hardness, float resistance) {
		delegate.method_9629(hardness, resistance);
		return this;
	}

	public FabricBlockSettings breakInstantly() {
		BlockSettingsExtensions.breakInstantly(delegate);
		return this;
	}

	public FabricBlockSettings dropsNothing() {
		BlockSettingsExtensions.dropsNothing(delegate);
		return this;
	}

	public FabricBlockSettings dropsLike(class_2248 block) {
		delegate.method_16228(block);
		return this;
	}

	public FabricBlockSettings drops(class_2960 dropTableId) {
		BlockSettingsExtensions.drops(delegate, dropTableId);
		return this;
	}

	@Deprecated
	public FabricBlockSettings friction(float friction) {
		delegate.method_9628(friction);
		return this;
	}

	public FabricBlockSettings slipperiness(float value) {
		delegate.method_9628(value);
		return this;
	}

	public FabricBlockSettings dynamicBounds() {
		BlockSettingsExtensions.dynamicBounds(delegate);
		return this;
	}

	/* BUILDING LOGIC */

	public class_2248.Settings build() {
		return delegate;
	}

	public <T> T build(Function<class_2248.Settings, T> function) {
		return function.apply(delegate);
	}
}
