/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.recipe.ingredient.builtin;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.serialization.JsonOps;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredient;
import net.fabricmc.fabric.api.recipe.v1.ingredient.CustomIngredientSerializer;
import net.minecraft.class_1799;
import net.minecraft.class_1856;
import net.minecraft.class_2105;
import net.minecraft.class_2487;
import net.minecraft.class_2509;
import net.minecraft.class_2512;
import net.minecraft.class_2522;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3518;

public class NbtIngredient implements CustomIngredient {
	public static final CustomIngredientSerializer<NbtIngredient> SERIALIZER = new Serializer();

	private final class_1856 base;
	@Nullable
	private final class_2487 nbt;
	private final boolean strict;

	public NbtIngredient(class_1856 base, @Nullable class_2487 nbt, boolean strict) {
		if (nbt == null && !strict) {
			throw new IllegalArgumentException("NbtIngredient can only have null NBT in strict mode");
		}

		this.base = base;
		this.nbt = nbt;
		this.strict = strict;
	}

	@Override
	public boolean test(class_1799 stack) {
		if (!base.method_8093(stack)) return false;

		if (strict) {
			return Objects.equals(nbt, stack.method_7969());
		} else {
			return class_2512.method_10687(nbt, stack.method_7969(), true);
		}
	}

	@Override
	public List<class_1799> getMatchingStacks() {
		List<class_1799> stacks = new ArrayList<>(List.of(base.method_8105()));
		stacks.replaceAll(stack -> {
			class_1799 copy = stack.method_7972();

			if (nbt != null) {
				copy.method_7980(nbt.method_10553());
			}

			return copy;
		});
		stacks.removeIf(stack -> !base.method_8093(stack));
		return stacks;
	}

	@Override
	public boolean requiresTesting() {
		return true;
	}

	@Override
	public CustomIngredientSerializer<?> getSerializer() {
		return SERIALIZER;
	}

	private static class Serializer implements CustomIngredientSerializer<NbtIngredient> {
		private final Gson gson = new GsonBuilder().disableHtmlEscaping().create();
		private final class_2960 id = new class_2960("fabric", "nbt");

		@Override
		public class_2960 getIdentifier() {
			return id;
		}

		@Override
		public NbtIngredient read(JsonObject json) {
			class_1856 base = class_1856.method_52177(json.get("base"));
			class_2487 nbt = readNbt(json.get("nbt"));
			boolean strict = class_3518.method_15258(json, "strict", false);
			return new NbtIngredient(base, nbt, strict);
		}

		/**
		 * Inspiration taken from {@link class_2105#method_9073}.
		 */
		@Nullable
		private static class_2487 readNbt(@Nullable JsonElement json) {
			// Process null
			if (json == null || json.isJsonNull()) {
				return null;
			}

			try {
				if (json.isJsonObject()) {
					// We use a normal .toString() to convert the json to string, and read it as SNBT.
					// Using DynamicOps would mess with the type of integers and cause things like damage comparisons to fail...
					return class_2522.method_10718(json.toString());
				} else {
					// Assume it's a string representation of the NBT
					return class_2522.method_10718(class_3518.method_15287(json, "nbt"));
				}
			} catch (CommandSyntaxException commandSyntaxException) {
				throw new JsonSyntaxException("Invalid nbt tag: " + commandSyntaxException.getMessage());
			}
		}

		@Override
		public void write(JsonObject json, NbtIngredient ingredient) {
			json.add("base", ingredient.base.method_8089());
			json.addProperty("strict", ingredient.strict);

			if (ingredient.nbt != null) {
				json.add("nbt", class_2509.field_11560.method_29146(JsonOps.INSTANCE, ingredient.nbt));
			}
		}

		@Override
		public NbtIngredient read(class_2540 buf) {
			class_1856 base = class_1856.method_8086(buf);
			class_2487 nbt = buf.method_10798();
			boolean strict = buf.readBoolean();
			return new NbtIngredient(base, nbt, strict);
		}

		@Override
		public void write(class_2540 buf, NbtIngredient ingredient) {
			ingredient.base.method_8088(buf);
			buf.method_10794(ingredient.nbt);
			buf.writeBoolean(ingredient.strict);
		}
	}
}
