/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.registry.sync;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;

import com.google.common.base.Joiner;
import com.google.common.collect.Sets;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.ints.IntSet;
import it.unimi.dsi.fastutil.objects.Object2IntLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2IntMap;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.minecraft.class_124;
import net.minecraft.class_1255;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2561;
import net.minecraft.class_2960;
import net.minecraft.class_3222;
import net.minecraft.class_5250;
import net.minecraft.class_5321;
import net.minecraft.class_7923;
import net.minecraft.server.MinecraftServer;
import net.fabricmc.fabric.api.event.registry.RegistryAttribute;
import net.fabricmc.fabric.api.event.registry.RegistryAttributeHolder;
import net.fabricmc.fabric.impl.registry.sync.packet.DirectRegistryPacketHandler;
import net.fabricmc.fabric.impl.registry.sync.packet.RegistryPacketHandler;

public final class RegistrySyncManager {
	public static final boolean DEBUG = Boolean.getBoolean("fabric.registry.debug");

	public static final RegistryPacketHandler DIRECT_PACKET_HANDLER = new DirectRegistryPacketHandler();

	private static final Logger LOGGER = LoggerFactory.getLogger("FabricRegistrySync");
	private static final boolean DEBUG_WRITE_REGISTRY_DATA = Boolean.getBoolean("fabric.registry.debug.writeContentsAsCsv");

	//Set to true after vanilla's bootstrap has completed
	public static boolean postBootstrap = false;

	private RegistrySyncManager() { }

	public static void sendPacket(MinecraftServer server, class_3222 player) {
		if (!DEBUG && server.method_19466(player.method_7334())) {
			return;
		}

		sendPacket(player, DIRECT_PACKET_HANDLER);
	}

	private static void sendPacket(class_3222 player, RegistryPacketHandler handler) {
		Map<class_2960, Object2IntMap<class_2960>> map = RegistrySyncManager.createAndPopulateRegistryMap(true, null);

		if (map != null) {
			handler.sendPacket(player, map);
		}
	}

	public static void receivePacket(class_1255<?> executor, RegistryPacketHandler handler, class_2540 buf, boolean accept, Consumer<Exception> errorHandler) {
		handler.receivePacket(buf);

		if (!handler.isPacketFinished()) {
			return;
		}

		if (DEBUG) {
			String handlerName = handler.getClass().getSimpleName();
			LOGGER.info("{} total packet: {}", handlerName, handler.getTotalPacketReceived());
			LOGGER.info("{} raw size: {}", handlerName, handler.getRawBufSize());
			LOGGER.info("{} deflated size: {}", handlerName, handler.getDeflatedBufSize());
		}

		Map<class_2960, Object2IntMap<class_2960>> map = handler.getSyncedRegistryMap();

		if (accept) {
			try {
				executor.method_5385(() -> {
					if (map == null) {
						errorHandler.accept(new RemapException("Received null map in sync packet!"));
						return null;
					}

					try {
						apply(map, RemappableRegistry.RemapMode.REMOTE);
					} catch (RemapException e) {
						errorHandler.accept(e);
					}

					return null;
				}).get(30, TimeUnit.SECONDS);
			} catch (ExecutionException | InterruptedException | TimeoutException e) {
				errorHandler.accept(e);
			}
		}
	}

	/**
	 * Creates a {@link class_2487} used to save or sync the registry ids.
	 *
	 * @param isClientSync true when syncing to the client, false when saving
	 * @param activeMap    contains the registry ids that were previously read and applied, can be null.
	 * @return a {@link class_2487} to save or sync, null when empty
	 */
	@Nullable
	public static Map<class_2960, Object2IntMap<class_2960>> createAndPopulateRegistryMap(boolean isClientSync, @Nullable Map<class_2960, Object2IntMap<class_2960>> activeMap) {
		Map<class_2960, Object2IntMap<class_2960>> map = new LinkedHashMap<>();

		for (class_2960 registryId : class_7923.field_41167.method_10235()) {
			class_2378 registry = class_7923.field_41167.method_10223(registryId);

			if (DEBUG_WRITE_REGISTRY_DATA) {
				File location = new File(".fabric" + File.separatorChar + "debug" + File.separatorChar + "registry");
				boolean c = true;

				if (!location.exists()) {
					if (!location.mkdirs()) {
						LOGGER.warn("[fabric-registry-sync debug] Could not create " + location.getAbsolutePath() + " directory!");
						c = false;
					}
				}

				if (c && registry != null) {
					File file = new File(location, registryId.toString().replace(':', '.').replace('/', '.') + ".csv");

					try (FileOutputStream stream = new FileOutputStream(file)) {
						StringBuilder builder = new StringBuilder("Raw ID,String ID,Class Type\n");

						for (Object o : registry) {
							String classType = (o == null) ? "null" : o.getClass().getName();
							//noinspection unchecked
							class_2960 id = registry.method_10221(o);
							if (id == null) continue;

							//noinspection unchecked
							int rawId = registry.method_10206(o);
							String stringId = id.toString();
							builder.append("\"").append(rawId).append("\",\"").append(stringId).append("\",\"").append(classType).append("\"\n");
						}

						stream.write(builder.toString().getBytes(StandardCharsets.UTF_8));
					} catch (IOException e) {
						LOGGER.warn("[fabric-registry-sync debug] Could not write to " + file.getAbsolutePath() + "!", e);
					}
				}
			}

			/*
			 * This contains the previous state's registry data, this is used for a few things:
			 * Such as ensuring that previously modded registries or registry entries are not lost or overwritten.
			 */
			Object2IntMap<class_2960> previousIdMap = null;

			if (activeMap != null && activeMap.containsKey(registryId)) {
				previousIdMap = activeMap.get(registryId);
			}

			RegistryAttributeHolder attributeHolder = RegistryAttributeHolder.get(registry.method_30517());

			if (!attributeHolder.hasAttribute(isClientSync ? RegistryAttribute.SYNCED : RegistryAttribute.PERSISTED)) {
				LOGGER.debug("Not {} registry: {}", isClientSync ? "syncing" : "saving", registryId);
				continue;
			}

			/*
			 * Dont do anything with vanilla registries on client sync.
			 * When saving skip none modded registries that doesnt have previous registry data
			 *
			 * This will not sync IDs if a world has been previously modded, either from removed mods
			 * or a previous version of fabric registry sync, but will save these ids to disk in case the mod or mods
			 * are added back.
			 */
			if ((previousIdMap == null || isClientSync) && !attributeHolder.hasAttribute(RegistryAttribute.MODDED)) {
				LOGGER.debug("Skipping un-modded registry: " + registryId);
				continue;
			} else if (previousIdMap != null) {
				LOGGER.debug("Preserving previously modded registry: " + registryId);
			}

			if (isClientSync) {
				LOGGER.debug("Syncing registry: " + registryId);
			} else {
				LOGGER.debug("Saving registry: " + registryId);
			}

			if (registry instanceof RemappableRegistry) {
				Object2IntMap<class_2960> idMap = new Object2IntLinkedOpenHashMap<>();
				IntSet rawIdsFound = DEBUG ? new IntOpenHashSet() : null;

				for (Object o : registry) {
					//noinspection unchecked
					class_2960 id = registry.method_10221(o);
					if (id == null) continue;

					//noinspection unchecked
					int rawId = registry.method_10206(o);

					if (DEBUG) {
						if (registry.method_10223(id) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> string ID " + id + " -> object " + registry.method_10223(id) + "!");
						}

						if (registry.method_10200(rawId) != o) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": object " + o + " -> integer ID " + rawId + " -> object " + registry.method_10200(rawId) + "!");
						}

						if (!rawIdsFound.add(rawId)) {
							LOGGER.error("[fabric-registry-sync] Inconsistency detected in " + registryId + ": multiple objects hold the raw ID " + rawId + " (this one is " + id + ")");
						}
					}

					idMap.put(id, rawId);
				}

				/*
				 * Look for existing registry key/values that are not in the current registries.
				 * This can happen when registry entries are removed, preventing that ID from being re-used by something else.
				 */
				if (!isClientSync && previousIdMap != null) {
					for (class_2960 key : previousIdMap.keySet()) {
						if (!idMap.containsKey(key)) {
							LOGGER.debug("Saving orphaned registry entry: " + key);
							idMap.put(key, previousIdMap.getInt(key));
						}
					}
				}

				map.put(registryId, idMap);
			}
		}

		// Ensure any orphaned registry's are kept on disk
		if (!isClientSync && activeMap != null) {
			for (class_2960 registryKey : activeMap.keySet()) {
				if (!map.containsKey(registryKey)) {
					LOGGER.debug("Saving orphaned registry: " + registryKey);
					map.put(registryKey, activeMap.get(registryKey));
				}
			}
		}

		if (map.isEmpty()) {
			return null;
		}

		return map;
	}

	public static void apply(Map<class_2960, Object2IntMap<class_2960>> map, RemappableRegistry.RemapMode mode) throws RemapException {
		if (mode == RemappableRegistry.RemapMode.REMOTE) {
			checkRemoteRemap(map);
		}

		Set<class_2960> containedRegistries = Sets.newHashSet(map.keySet());

		for (class_2960 registryId : class_7923.field_41167.method_10235()) {
			if (!containedRegistries.remove(registryId)) {
				continue;
			}

			Object2IntMap<class_2960> registryMap = map.get(registryId);
			class_2378<?> registry = class_7923.field_41167.method_10223(registryId);

			RegistryAttributeHolder attributeHolder = RegistryAttributeHolder.get(registry.method_30517());

			if (!attributeHolder.hasAttribute(RegistryAttribute.MODDED)) {
				LOGGER.debug("Not applying registry data to vanilla registry {}", registryId.toString());
				continue;
			}

			if (registry instanceof RemappableRegistry) {
				Object2IntMap<class_2960> idMap = new Object2IntOpenHashMap<>();

				for (class_2960 key : registryMap.keySet()) {
					idMap.put(key, registryMap.getInt(key));
				}

				((RemappableRegistry) registry).remap(registryId.toString(), idMap, mode);
			}
		}

		if (!containedRegistries.isEmpty()) {
			LOGGER.warn("[fabric-registry-sync] Could not find the following registries: " + Joiner.on(", ").join(containedRegistries));
		}
	}

	@VisibleForTesting
	public static void checkRemoteRemap(Map<class_2960, Object2IntMap<class_2960>> map) throws RemapException {
		Map<class_2960, List<class_2960>> missingEntries = new HashMap<>();

		for (Map.Entry<? extends class_5321<? extends class_2378<?>>, ? extends class_2378<?>> entry : class_7923.field_41167.method_29722()) {
			final class_2378<?> registry = entry.getValue();
			final class_2960 registryId = entry.getKey().method_29177();
			final Object2IntMap<class_2960> remoteRegistry = map.get(registryId);

			if (remoteRegistry == null) {
				// Registry sync does not contain data for this registry, will print a warning when applying.
				continue;
			}

			for (class_2960 remoteId : remoteRegistry.keySet()) {
				if (!registry.method_10250(remoteId)) {
					// Found a registry entry from the server that is
					missingEntries.computeIfAbsent(registryId, i -> new ArrayList<>()).add(remoteId);
				}
			}
		}

		if (missingEntries.isEmpty()) {
			// All good :)
			return;
		}

		// Print out details to the log
		LOGGER.error("Received unknown remote registry entries from server");

		for (Map.Entry<class_2960, List<class_2960>> entry : missingEntries.entrySet()) {
			for (class_2960 identifier : entry.getValue()) {
				LOGGER.error("Registry entry ({}) is missing from local registry ({})", identifier, entry.getKey());
			}
		}

		// Create a nice user friendly error message.
		class_5250 text = class_2561.method_43470("");

		final int count = missingEntries.values().stream().mapToInt(List::size).sum();

		if (count == 1) {
			text = text.method_10852(class_2561.method_43471("fabric-registry-sync-v0.unknown-remote.title.singular"));
		} else {
			text = text.method_10852(class_2561.method_43469("fabric-registry-sync-v0.unknown-remote.title.plural", count));
		}

		text = text.method_10852(class_2561.method_43471("fabric-registry-sync-v0.unknown-remote.subtitle.1").method_27692(class_124.field_1060));
		text = text.method_10852(class_2561.method_43471("fabric-registry-sync-v0.unknown-remote.subtitle.2"));

		final int toDisplay = 4;
		// Get the distinct missing namespaces
		final List<String> namespaces = missingEntries.values().stream()
				.flatMap(List::stream)
				.map(class_2960::method_12836)
				.distinct()
				.sorted()
				.toList();

		for (int i = 0; i < Math.min(namespaces.size(), toDisplay); i++) {
			text = text.method_10852(class_2561.method_43470(namespaces.get(i)).method_27692(class_124.field_1054));
			text = text.method_27693("\n");
		}

		if (namespaces.size() > toDisplay) {
			text = text.method_10852(class_2561.method_43469("fabric-registry-sync-v0.unknown-remote.footer", namespaces.size() - toDisplay));
		}

		throw new RemapException(text);
	}

	public static void unmap() throws RemapException {
		for (class_2960 registryId : class_7923.field_41167.method_10235()) {
			class_2378 registry = class_7923.field_41167.method_10223(registryId);

			if (registry instanceof RemappableRegistry) {
				((RemappableRegistry) registry).unmap(registryId.toString());
			}
		}
	}

	public static void bootstrapRegistries() {
		postBootstrap = true;
	}
}
