/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.renderer;

import java.util.List;
import java.util.function.Supplier;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.renderer.v1.Renderer;
import net.fabricmc.fabric.api.renderer.v1.RendererAccess;
import net.fabricmc.fabric.api.renderer.v1.material.RenderMaterial;
import net.fabricmc.fabric.api.renderer.v1.mesh.QuadEmitter;
import net.fabricmc.fabric.api.renderer.v1.model.ModelHelper;
import net.fabricmc.fabric.api.renderer.v1.render.RenderContext;
import net.fabricmc.fabric.api.util.TriState;
import net.minecraft.class_1087;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_5819;
import net.minecraft.class_777;

/**
 * Routines for adaptation of vanilla {@link class_1087}s to FRAPI pipelines.
 * Even though Indigo calls them directly, they are not for use by third party renderers, and might change at any time.
 */
public class VanillaModelEncoder {
	private static final Renderer RENDERER = RendererAccess.INSTANCE.getRenderer();
	private static final RenderMaterial MATERIAL_STANDARD = RENDERER.materialFinder().find();
	private static final RenderMaterial MATERIAL_NO_AO = RENDERER.materialFinder().ambientOcclusion(TriState.FALSE).find();

	// Separate QuadEmitter parameter so that Indigo can pass its own emitter that handles vanilla quads differently.
	public static void emitBlockQuads(class_1087 model, @Nullable class_2680 state, Supplier<class_5819> randomSupplier, RenderContext context, QuadEmitter emitter) {
		final RenderMaterial defaultMaterial = model.method_4708() ? MATERIAL_STANDARD : MATERIAL_NO_AO;

		for (int i = 0; i <= ModelHelper.NULL_FACE_ID; i++) {
			final class_2350 cullFace = ModelHelper.faceFromIndex(i);

			if (!context.hasTransform() && context.isFaceCulled(cullFace)) {
				// Skip entire quad list if possible.
				continue;
			}

			final List<class_777> quads = model.method_4707(state, cullFace, randomSupplier.get());
			final int count = quads.size();

			for (int j = 0; j < count; j++) {
				final class_777 q = quads.get(j);
				emitter.fromVanilla(q, defaultMaterial, cullFace);
				emitter.emit();
			}
		}
	}

	public static void emitItemQuads(class_1087 model, @Nullable class_2680 state, Supplier<class_5819> randomSupplier, RenderContext context) {
		QuadEmitter emitter = context.getEmitter();

		for (int i = 0; i <= ModelHelper.NULL_FACE_ID; i++) {
			final class_2350 cullFace = ModelHelper.faceFromIndex(i);
			final List<class_777> quads = model.method_4707(state, cullFace, randomSupplier.get());
			final int count = quads.size();

			for (int j = 0; j < count; j++) {
				final class_777 q = quads.get(j);
				emitter.fromVanilla(q, MATERIAL_STANDARD, cullFace);
				emitter.emit();
			}
		}
	}
}
