/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.renderer.v1;

import java.util.List;
import java.util.function.Predicate;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.renderer.v1.material.MaterialFinder;
import net.fabricmc.fabric.api.renderer.v1.material.RenderMaterial;
import net.fabricmc.fabric.api.renderer.v1.mesh.MutableMesh;
import net.fabricmc.fabric.api.renderer.v1.mesh.QuadEmitter;
import net.fabricmc.fabric.api.renderer.v1.model.FabricBlockModelPart;
import net.fabricmc.fabric.api.renderer.v1.render.FabricBlockModelRenderer;
import net.fabricmc.fabric.api.renderer.v1.render.FabricBlockRenderManager;
import net.fabricmc.fabric.api.renderer.v1.render.FabricLayerRenderState;
import net.fabricmc.fabric.impl.renderer.RendererManager;
import net.fabricmc.fabric.impl.renderer.VanillaBlockModelPartEncoder;
import net.minecraft.class_10444;
import net.minecraft.class_1087;
import net.minecraft.class_10889;
import net.minecraft.class_1920;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2680;
import net.minecraft.class_2960;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_5819;
import net.minecraft.class_776;
import net.minecraft.class_778;
import net.minecraft.class_9810;

/**
 * Interface for rendering plug-ins that provide enhanced capabilities
 * for model lighting, buffering and rendering. Such plug-ins implement the
 * enhanced model rendering interfaces specified by the Fabric API.
 *
 * <p>Renderers must ensure that terrain buffering supports {@link class_1087#emitQuads}, which happens in
 * {@link class_9810} in vanilla; this code is not patched automatically. Renderers must also ensure that the
 * following vanilla methods support {@link class_1087#emitQuads}; these methods are not patched automatically.
 *
 * <ul><li>{@link class_778#method_3367(class_4587.class_4665, class_4588, class_1087, float, float, float, int, int)}
 *
 * <li>{@link class_776#method_23071(class_2680, class_2338, class_1920, class_4587, class_4588)}
 *
 * <li>{@link class_776#method_3353(class_2680, class_4587, class_4597, int, int)}</ul>
 *
 * <p>All other places in vanilla code that invoke {@link class_1087#method_68513(class_5819, List)},
 * {@link class_1087#method_68512(class_5819)}, or
 * {@link class_778#method_3367(class_4587.class_4665, class_4588, class_1087, float, float, float, int, int)}
 * are, where appropriate, patched automatically to invoke the corresponding method above or the corresponding method in
 * {@link FabricBlockModelRenderer} or {@link FabricBlockRenderManager}.
 */
public interface Renderer {
	/**
	 * Access to the current {@link Renderer} for creating and retrieving mesh builders
	 * and materials.
	 */
	static Renderer get() {
		return RendererManager.getRenderer();
	}

	/**
	 * Rendering extension mods must implement {@link Renderer} and
	 * call this method during initialization.
	 *
	 * <p>Only one {@link Renderer} plug-in can be active in any game instance.
	 * If a second mod attempts to register, this method will throw an UnsupportedOperationException.
	 */
	static void register(Renderer renderer) {
		RendererManager.registerRenderer(renderer);
	}

	/**
	 * Obtain a new {@link MutableMesh} instance to build optimized meshes and create baked models
	 * with enhanced features.
	 *
	 * <p>Renderer does not retain a reference to returned instances, so they should be re-used
	 * when possible to avoid memory allocation overhead.
	 */
	MutableMesh mutableMesh();

	/**
	 * Obtain a new {@link MaterialFinder} instance to retrieve standard {@link RenderMaterial}
	 * instances.
	 *
	 * <p>Renderer does not retain a reference to returned instances, so they should be re-used for
	 * multiple materials when possible to avoid memory allocation overhead.
	 */
	MaterialFinder materialFinder();

	/**
	 * Return a material previously registered via {@link #registerMaterial(class_2960, RenderMaterial)}.
	 * Will return null if no material was found matching the given identifier.
	 */
	@Nullable
	RenderMaterial materialById(class_2960 id);

	/**
	 * Register a material for re-use by other mods or models within a mod.
	 * The registry does not persist registrations - mods must create and register
	 * all materials at game initialization.
	 *
	 * <p>Returns false if a material with the given identifier is already present,
	 * leaving the existing material intact.
	 */
	boolean registerMaterial(class_2960 id, RenderMaterial material);

	/**
	 * @see FabricBlockModelRenderer#render(class_1920, class_1087, class_2680, class_2338, class_4587, class_4597, boolean, long, int)
	 */
	@ApiStatus.OverrideOnly
	void render(class_778 modelRenderer, class_1920 blockView, class_1087 model, class_2680 state, class_2338 pos, class_4587 matrices, class_4597 vertexConsumers, boolean cull, long seed, int overlay);

	/**
	 * @see FabricBlockModelRenderer#render(class_4587.class_4665, class_4597, class_1087, float, float, float, int, int, class_1920, class_2338, class_2680)
	 */
	@ApiStatus.OverrideOnly
	void render(class_4587.class_4665 matrices, class_4597 vertexConsumers, class_1087 model, float red, float green, float blue, int light, int overlay, class_1920 blockView, class_2338 pos, class_2680 state);

	/**
	 * @see FabricBlockRenderManager#renderBlockAsEntity(class_2680, class_4587, class_4597, int, int, class_1920, class_2338)
	 */
	@ApiStatus.OverrideOnly
	void renderBlockAsEntity(class_776 renderManager, class_2680 state, class_4587 matrices, class_4597 vertexConsumers, int light, int overlay, class_1920 blockView, class_2338 pos);

	/**
	 * @see FabricBlockModelPart#emitQuads(QuadEmitter, Predicate)
	 */
	@ApiStatus.Experimental
	@ApiStatus.OverrideOnly
	default void emitBlockModelPartQuads(class_10889 modelPart, QuadEmitter emitter, Predicate<@Nullable class_2350> cullTest) {
		VanillaBlockModelPartEncoder.emitQuads(modelPart, emitter, cullTest);
	}

	/**
	 * @see FabricLayerRenderState#emitter()
	 */
	@ApiStatus.OverrideOnly
	QuadEmitter getLayerRenderStateEmitter(class_10444.class_10446 layer);
}
