/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.renderer.v1.model;

import java.util.EnumMap;
import java.util.Map;
import org.joml.Matrix3f;
import org.joml.Matrix4f;
import org.joml.Matrix4fc;
import org.joml.Vector3f;
import org.joml.Vector4f;
import net.fabricmc.fabric.api.renderer.v1.mesh.QuadTransform;
import net.fabricmc.fabric.api.renderer.v1.sprite.SpriteFinderGetter;
import net.minecraft.class_1058;
import net.minecraft.class_10820;
import net.minecraft.class_1086;
import net.minecraft.class_2350;
import net.minecraft.class_3665;
import net.minecraft.class_4590;
import net.minecraft.class_4609;
import net.minecraft.class_7837;

/**
 * Utilities to make it easier to work with {@link class_3665}.
 */
public final class ModelBakeSettingsHelper {
	private static final class_2350[] DIRECTIONS = class_2350.values();

	private ModelBakeSettingsHelper() {
	}

	/**
	 * Creates a new {@link class_3665} using the given transformation and enables UV lock if specified. Works
	 * exactly like {@link class_1086}, but allows an arbitrary transformation. Instances should be retained and
	 * reused, especially if UV lock is enabled, to avoid redoing costly computations.
	 */
	public static class_3665 of(class_4590 transformation, boolean uvLock) {
		Matrix4fc matrix = transformation.method_22936();

		if (class_7837.method_65174(matrix)) {
			return class_1086.field_63619;
		}

		if (!uvLock) {
			return new class_3665() {
				@Override
				public class_4590 method_3509() {
					return transformation;
				}
			};
		}

		Map<class_2350, Matrix4fc> faceTransformations = new EnumMap<>(class_2350.class);
		Map<class_2350, Matrix4fc> inverseFaceTransformations = new EnumMap<>(class_2350.class);

		for (class_2350 face : DIRECTIONS) {
			Matrix4fc faceTransformation = class_4609.method_68069(transformation, face).method_22936();
			faceTransformations.put(face, faceTransformation);
			inverseFaceTransformations.put(face, faceTransformation.invert(new Matrix4f()));
		}

		return new class_3665() {
			@Override
			public class_4590 method_3509() {
				return transformation;
			}

			@Override
			public Matrix4fc method_68011(class_2350 face) {
				return faceTransformations.get(face);
			}

			@Override
			public Matrix4fc method_68012(class_2350 face) {
				return inverseFaceTransformations.get(face);
			}
		};
	}

	/**
	 * Creates a new {@link class_3665} that is the product of the two given settings. Settings are represented
	 * by matrices, so this method follows the rules of matrix multiplication, namely that applying the resulting
	 * settings is (mostly) equivalent to applying the right settings and then the left settings. The only exception
	 * during standard application is cull face transformation, as the result must be clamped. Thus, applying a single
	 * premultiplied transformation generally yields better results than multiple applications.
	 */
	public static class_3665 multiply(class_3665 left, class_3665 right) {
		// Assumes face transformations are identity if main transformation is identity
		if (class_7837.method_65174(left.method_3509().method_22936())) {
			return right;
		} else if (class_7837.method_65174(right.method_3509().method_22936())) {
			return left;
		}

		class_4590 transformation = left.method_3509().method_22933(right.method_3509());

		boolean leftHasFaceTransformations = false;
		boolean rightHasFaceTransformations = false;

		// Assumes inverse face transformations are exactly inverse of regular face transformations
		for (class_2350 face : DIRECTIONS) {
			if (!leftHasFaceTransformations && !class_7837.method_65174(left.method_68011(face))) {
				leftHasFaceTransformations = true;
			}

			if (!rightHasFaceTransformations && !class_7837.method_65174(right.method_68011(face))) {
				rightHasFaceTransformations = true;
			}
		}

		if (leftHasFaceTransformations & rightHasFaceTransformations) {
			Map<class_2350, Matrix4fc> faceTransformations = new EnumMap<>(class_2350.class);
			Map<class_2350, Matrix4fc> inverseFaceTransformations = new EnumMap<>(class_2350.class);

			for (class_2350 face : DIRECTIONS) {
				faceTransformations.put(face, left.method_68011(face).mul(right.method_68011(face), new Matrix4f()));
				inverseFaceTransformations.put(face, right.method_68012(face).mul(left.method_68012(face), new Matrix4f()));
			}

			return new class_3665() {
				@Override
				public class_4590 method_3509() {
					return transformation;
				}

				@Override
				public Matrix4fc method_68011(class_2350 face) {
					return faceTransformations.get(face);
				}

				@Override
				public Matrix4fc method_68012(class_2350 face) {
					return inverseFaceTransformations.get(face);
				}
			};
		}

		class_3665 faceTransformDelegate = leftHasFaceTransformations ? left : right;

		return new class_3665() {
			@Override
			public class_4590 method_3509() {
				return transformation;
			}

			@Override
			public Matrix4fc method_68011(class_2350 face) {
				return faceTransformDelegate.method_68011(face);
			}

			@Override
			public Matrix4fc method_68012(class_2350 face) {
				return faceTransformDelegate.method_68012(face);
			}
		};
	}

	/**
	 * Creates a new {@link QuadTransform} that applies the given transformation. The sprite finder is used to look up
	 * the current sprite to correctly apply UV lock, if present in the transformation.
	 *
	 * <p>This method is most useful when creating custom implementations of {@link class_10820}, which receive a
	 * {@link class_3665}.
	 */
	public static QuadTransform asQuadTransform(class_3665 settings, SpriteFinderGetter spriteFinderGetter) {
		Matrix4fc matrix = settings.method_3509().method_22936();

		// Assumes face transformations are identity if main transformation is identity
		if (class_7837.method_65174(matrix)) {
			return q -> true;
		}

		Matrix3f normalMatrix = matrix.normal(new Matrix3f());

		Vector4f vec4 = new Vector4f();
		Vector3f vec3 = new Vector3f();

		return quad -> {
			class_2350 lightFace = quad.lightFace();
			Matrix4fc reverseMatrix = settings.method_68012(lightFace);

			if (!class_7837.method_65174(reverseMatrix)) {
				SpriteFinder spriteFinder = spriteFinderGetter.spriteFinder(quad.atlas());
				class_1058 sprite = spriteFinder.find(quad);

				for (int vertexIndex = 0; vertexIndex < 4; vertexIndex++) {
					float frameU = getFrameFromU(sprite, quad.u(vertexIndex));
					float frameV = getFrameFromV(sprite, quad.v(vertexIndex));
					vec3.set(frameU - 0.5f, frameV - 0.5f, 0.0f);
					reverseMatrix.transformPosition(vec3);
					frameU = vec3.x + 0.5f;
					frameV = vec3.y + 0.5f;
					quad.uv(vertexIndex, sprite.method_4580(frameU), sprite.method_4570(frameV));
				}
			}

			for (int vertexIndex = 0; vertexIndex < 4; vertexIndex++) {
				vec4.set(quad.x(vertexIndex) - 0.5f, quad.y(vertexIndex) - 0.5f, quad.z(vertexIndex) - 0.5f, 1.0f);
				vec4.mul(matrix);
				quad.pos(vertexIndex, vec4.x + 0.5f, vec4.y + 0.5f, vec4.z + 0.5f);

				if (quad.hasNormal(vertexIndex)) {
					quad.copyNormal(vertexIndex, vec3);
					vec3.mul(normalMatrix);
					vec3.normalize();
					quad.normal(vertexIndex, vec3);
				}
			}

			class_2350 cullFace = quad.cullFace();

			if (cullFace != null) {
				quad.cullFace(class_2350.method_23225(matrix, cullFace));
			}

			return true;
		};
	}

	private static float getFrameFromU(class_1058 sprite, float u) {
		float f = sprite.method_4577() - sprite.method_4594();
		return (u - sprite.method_4594()) / f;
	}

	private static float getFrameFromV(class_1058 sprite, float v) {
		float f = sprite.method_4575() - sprite.method_4593();
		return (v - sprite.method_4593()) / f;
	}
}
