/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.indigo.renderer.helper;

import static net.minecraft.util.math.MathHelper.equalsApproximate;

import net.fabricmc.fabric.api.renderer.v1.mesh.QuadView;
import net.minecraft.class_1160;
import net.minecraft.class_2350;
import net.minecraft.class_2350.class_2351;
import net.minecraft.class_2350.class_2352;

/**
 * Static routines of general utility for renderer implementations.
 * Renderers are not required to use these helpers, but they were
 * designed to be usable without the default renderer.
 */
public abstract class GeometryHelper {
    /** set when a quad touches all four corners of a unit cube */
    public static final int CUBIC_FLAG = 1;
    
    /** set when a quad is parallel to (but not necessarily on) a its light face */
    public static final int AXIS_ALIGNED_FLAG = CUBIC_FLAG << 1;
    
    /** set when a quad is coplanar with its light face. Implies {@link #AXIS_ALIGNED_FLAG} */
    public static final int LIGHT_FACE_FLAG = AXIS_ALIGNED_FLAG << 1;
    
    private GeometryHelper() {}

    /**
     * Analyzes the quad and returns a value with some combination 
     * of {@link #AXIS_ALIGNED_FLAG}, {@link #LIGHT_FACE_FLAG} and {@link #CUBIC_FLAG}.
     * Intended use is to optimize lighting when the geometry is regular.
     * Expects convex quads with all points co-planar.
     */
    public static int computeShapeFlags(QuadView quad) {
        class_2350 lightFace = quad.lightFace();
        int bits = 0;
        if(isQuadParallelToFace(lightFace, quad)) {
            bits |= AXIS_ALIGNED_FLAG;
            if(isParallelQuadOnFace(lightFace, quad)) {
                bits |= LIGHT_FACE_FLAG;
            }
            if(isQuadCubic(lightFace, quad)) {
                bits |= CUBIC_FLAG;
            }
        }
        return bits;
    }
    
    /**
     * Returns true if quad is parallel to the given face.
     * Does not validate quad winding order.
     * Expects convex quads with all points co-planar.
     */
    public static boolean isQuadParallelToFace(class_2350 face, QuadView quad) {
        if(face == null) {
            return false;
        }
        int i = face.method_10166().ordinal();
        final float val = quad.posByIndex(0, i);
        return method_15347(val, quad.posByIndex(1, i))
                && method_15347(val, quad.posByIndex(2, i))
                && method_15347(val, quad.posByIndex(3, i));
    }
    
    /**
     * True if quad - already known to be parallel to a face - is actually coplanar with it.<p>
     * 
     * Test will be unreliable if not already parallel, use {@link #isQuadParallel(Direction, QuadView)}
     * for that purpose. Expects convex quads with all points co-planar.<p>
     */
    public static boolean isParallelQuadOnFace(class_2350 lightFace, QuadView quad) {
        if(lightFace == null)
            return false;
        final int coordinateIndex = lightFace.method_10166().ordinal();
        final float expectedValue = lightFace.method_10171() == class_2352.field_11056 ? 1 : 0;
        return method_15347(quad.posByIndex(0, coordinateIndex), expectedValue);
    }
    
    /**
     * Returns true if quad is truly a quad (not a triangle) and fills a full block cross-section.
     * If known to be true, allows use of a simpler/faster AO lighting algorithm.<p>
     * 
     * Does not check if quad is actually coplanar with the light face, nor does it check that all
     * quad vertices are coplanar with each other. <p>
     * 
     * Expects convex quads with all points co-planar.<p>
     * 
     * @param lightFace MUST be non-null.
     */
    public static boolean isQuadCubic(class_2350 lightFace, QuadView quad) {
        if(lightFace == null) {
            return false;
        }
        
        int a, b;

        switch(lightFace) {
        case field_11034: 
        case field_11039:
            a = 1;
            b = 2;
            break;
        case field_11036:
        case field_11033:
            a = 0;
            b = 2;
            break;
        case field_11035:
        case field_11043:
            a = 1;
            b = 0;
            break;
        default:
            // handle WTF case
            return false;
        }
       
        return confirmSquareCorners(a, b, quad);
    }
    
    /**
     * Used by {@link #isQuadCubic(Direction, int[], int, QuadSerializer)}.
     * True if quad touches all four corners of unit square.
     */
    private static boolean confirmSquareCorners(int aCoordinate, int bCoordinate, QuadView quad) {
        int flags = 0;
        
        for(int i = 0; i < 4; i++) {
            final float a = quad.posByIndex(i, aCoordinate);
            final float b = quad.posByIndex(i, bCoordinate);
            
            if(method_15347(a, 0)) {
                if(method_15347(b, 0)) {
                    flags |= 1;
                } else if(method_15347(b, 1)) {
                    flags |= 2;
                } else {
                    return false;
                }
            } else if(method_15347(a, 1)) {
                if(method_15347(b, 0)) {
                    flags |= 4;
                } else if(method_15347(b, 1)) {
                    flags |= 8;
                } else {
                    return false;
                }
            } else {
                return false;
            }
        }
        return flags == 15;
    }
    
    /**
     * Identifies the face to which the quad is most closely aligned.
     * This mimics the value that {@link BakedQuad#getFace()} returns, and is
     * used in the vanilla renderer for all diffuse lighting.<p>
     * 
     * Derived from the quad face normal and expects convex quads with all points co-planar.
     */
    public static class_2350 lightFace(QuadView quad) {
        final class_1160 normal = quad.faceNormal();
        switch(GeometryHelper.longestAxis(normal)) {
            case field_11048:
                return normal.method_4943() > 0 ? class_2350.field_11034 : class_2350.field_11039;
                
            case field_11052:
                return normal.method_4945() > 0 ? class_2350.field_11036 : class_2350.field_11033;
                
            case field_11051:
                return normal.method_4947() > 0 ? class_2350.field_11035 : class_2350.field_11043;
            
            default:
                // handle WTF case
                return class_2350.field_11036;
        }
    }

    /**
     * Simple 4-way compare, doesn't handle NaN values.
     */
    public static float min(float a, float b, float c, float d) {
        final float x = a < b ? a : b;
        final float y = c < d ? c : d;
        return x < y ? x : y;
    }

    /**
     * Simple 4-way compare, doesn't handle NaN values.
     */
    public static float max(float a, float b, float c, float d) {
        final float x = a > b ? a : b;
        final float y = c > d ? c : d;
        return x > y ? x : y;
    }
    
    /**
     * See {@link #longestAxis(float, float, float)}
     */
    public static class_2351 longestAxis(class_1160 vec) {
        return longestAxis(vec.method_4943(), vec.method_4945(), vec.method_4947());
    }
    
    /**
     * Identifies the largest (max absolute magnitude) component (X, Y, Z) in the given vector.
     */
    public static class_2351 longestAxis(float normalX, float normalY, float normalZ) {
        class_2351 result = class_2351.field_11052;
        float longest = Math.abs(normalY);
    
        float a = Math.abs(normalX);
        if(a > longest)
        {
            result = class_2351.field_11048;
            longest = a;
        }
    
        return Math.abs(normalZ) > longest
                ? class_2351.field_11051 : result;
    }
}
