/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.indigo.renderer.render;

import static net.fabricmc.indigo.renderer.helper.GeometryHelper.LIGHT_FACE_FLAG;

import java.util.function.ToIntBiFunction;

import it.unimi.dsi.fastutil.ints.Int2ObjectFunction;
import net.fabricmc.fabric.api.renderer.v1.render.RenderContext.QuadTransform;
import net.fabricmc.indigo.renderer.accessor.AccessBufferBuilder;
import net.fabricmc.indigo.renderer.aocalc.AoCalculator;
import net.fabricmc.indigo.renderer.helper.ColorHelper;
import net.fabricmc.indigo.renderer.mesh.EncodingFormat;
import net.fabricmc.indigo.renderer.mesh.MutableQuadViewImpl;
import net.minecraft.class_2338;
import net.minecraft.class_2680;

/**
 * Base quad-rendering class for fallback and mesh consumers.
 * Has most of the actual buffer-time lighting and coloring logic.
 */
public abstract class AbstractQuadRenderer {
    private static final int FULL_BRIGHTNESS = 15 << 20 | 15 << 4;
    
    protected final ToIntBiFunction<class_2680, class_2338> brightnessFunc;
    protected final Int2ObjectFunction<AccessBufferBuilder> bufferFunc;
    protected final BlockRenderInfo blockInfo;
    protected final AoCalculator aoCalc;
    protected final QuadTransform transform;
    
    AbstractQuadRenderer(BlockRenderInfo blockInfo, ToIntBiFunction<class_2680, class_2338> brightnessFunc, Int2ObjectFunction<AccessBufferBuilder> bufferFunc, AoCalculator aoCalc, QuadTransform transform) {
        this.blockInfo = blockInfo;
        this.brightnessFunc = brightnessFunc;
        this.bufferFunc = bufferFunc;
        this.aoCalc = aoCalc;
        this.transform = transform;
    }
    
    
    /** handles block color and red-blue swizzle, common to all renders */
    private void colorizeQuad(MutableQuadViewImpl q, int blockColorIndex) {
        if(blockColorIndex == -1) {
            for(int i = 0; i < 4; i++) {
                q.spriteColor(i, 0, ColorHelper.swapRedBlueIfNeeded(q.spriteColor(i, 0)));
            }
        } else {
            final int blockColor = blockInfo.blockColor(blockColorIndex);
            for(int i = 0; i < 4; i++) {
                q.spriteColor(i, 0, ColorHelper.swapRedBlueIfNeeded(ColorHelper.multiplyColor(blockColor, q.spriteColor(i, 0))));
            }
        }
    }
    
    /** final output step, common to all renders */
    private void bufferQuad(MutableQuadViewImpl quad, int renderLayer) {
        bufferFunc.get(renderLayer).fabric_putVanillaData(quad.data(), quad.vertexStart());
    }

    // routines below have a bit of copy-paste code reuse to avoid conditional execution inside a hot loop
    
    /** for non-emissive mesh quads and all fallback quads with smooth lighting*/
    protected void tesselateSmooth(MutableQuadViewImpl q, int renderLayer, int blockColorIndex) {
        colorizeQuad(q, blockColorIndex);
        for(int i = 0; i < 4; i++) {
            q.spriteColor(i, 0, ColorHelper.multiplyRGB(q.spriteColor(i, 0), aoCalc.ao[i]));
            q.lightmap(i, ColorHelper.maxBrightness(q.lightmap(i), aoCalc.light[i]));
        }
        bufferQuad(q, renderLayer);
    }
    
    /** for emissive mesh quads with smooth lighting*/
    protected void tesselateSmoothEmissive(MutableQuadViewImpl q, int renderLayer, int blockColorIndex) {
        colorizeQuad(q, blockColorIndex);
        for(int i = 0; i < 4; i++) {
            q.spriteColor(i, 0, ColorHelper.multiplyRGB(q.spriteColor(i, 0), aoCalc.ao[i]));
            q.lightmap(i, FULL_BRIGHTNESS);
        }
        bufferQuad(q, renderLayer);
    }
    
    /** for non-emissive mesh quads and all fallback quads with flat lighting*/
    protected void tesselateFlat(MutableQuadViewImpl quad, int renderLayer, int blockColorIndex) {
        colorizeQuad(quad, blockColorIndex);
        final int brightness = flatBrightness(quad, blockInfo.blockState, blockInfo.blockPos);
        for(int i = 0; i < 4; i++) {
            quad.lightmap(i, ColorHelper.maxBrightness(quad.lightmap(i), brightness));
        }
        bufferQuad(quad, renderLayer);
    }
    
    /** for emissive mesh quads with flat lighting*/
    protected void tesselateFlatEmissive(MutableQuadViewImpl quad, int renderLayer, int blockColorIndex, int[] lightmaps) {
        colorizeQuad(quad, blockColorIndex);
        for(int i = 0; i < 4; i++) {
            quad.lightmap(i, FULL_BRIGHTNESS);
        }
        bufferQuad(quad, renderLayer);
    }
    
    protected int[] lightmaps = new int[4];
    
    protected void captureLightmaps(MutableQuadViewImpl q) {
        final int[] data = q.data();
        final int[] lightmaps = this.lightmaps;
        lightmaps[0] = data[EncodingFormat.VERTEX_START_OFFSET + 6];
        lightmaps[1] = data[EncodingFormat.VERTEX_START_OFFSET + 6 + 7];
        lightmaps[2] = data[EncodingFormat.VERTEX_START_OFFSET + 6 + 14];
        lightmaps[3] = data[EncodingFormat.VERTEX_START_OFFSET + 6 + 21];
    }
    
    protected void restoreLightmaps(MutableQuadViewImpl q) {
        final int[] data = q.data();
        final int[] lightmaps = this.lightmaps;
        data[EncodingFormat.VERTEX_START_OFFSET + 6] = lightmaps[0];
        data[EncodingFormat.VERTEX_START_OFFSET + 6 + 7] = lightmaps[1];
        data[EncodingFormat.VERTEX_START_OFFSET + 6 + 14] = lightmaps[2];
        data[EncodingFormat.VERTEX_START_OFFSET + 6 + 21] = lightmaps[3];
    }
    
    private final class_2338.class_2339 mpos = new class_2338.class_2339();
    
    /** 
     * Handles geometry-based check for using self brightness or neighbor brightness.
     * That logic only applies in flat lighting.
     */
    int flatBrightness(MutableQuadViewImpl quad, class_2680 blockState, class_2338 pos) {
        mpos.method_10101(pos);
        if((quad.geometryFlags() & LIGHT_FACE_FLAG) != 0) {
            mpos.method_10098(quad.lightFace());
        }
        return brightnessFunc.applyAsInt(blockState, mpos);
    }
}
