/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.indigo.renderer.render;

import it.unimi.dsi.fastutil.longs.Long2FloatOpenHashMap;
import it.unimi.dsi.fastutil.longs.Long2IntOpenHashMap;
import net.fabricmc.indigo.renderer.accessor.AccessBufferBuilder;
import net.fabricmc.indigo.renderer.accessor.AccessChunkRenderer;
import net.fabricmc.indigo.renderer.mesh.MutableQuadViewImpl;
import net.minecraft.class_1920;
import net.minecraft.class_1921;
import net.minecraft.class_2248.class_2250;
import net.minecraft.class_2338;
import net.minecraft.class_243;
import net.minecraft.class_2680;
import net.minecraft.class_287;
import net.minecraft.class_842;
import net.minecraft.class_849;
import net.minecraft.class_851;
import net.minecraft.class_853;

/**
 * Holds, manages and provides access to the chunk-related state
 * needed by fallback and mesh consumers during terrain rendering.<p>
 * 
 * Exception: per-block position offsets are tracked here so they can
 * be applied together with chunk offsets.
 */
public class ChunkRenderInfo {
    /**
     * Serves same function as brightness cache in Mojang's AO calculator,
     * with some differences as follows...<p>
     * 
     * 1) Mojang uses Object2Int.  This uses Long2Int for performance and to avoid
     * creating new immutable BlockPos references.  But will break if someone
     * wants to expand Y limit or world borders.  If we want to support that may
     * need to switch or make configurable.<p>
     * 
     * 2) Mojang overrides the map methods to limit the cache to 50 values.
     * However, a render chunk only has 18^3 blocks in it, and the cache is cleared every chunk.
     * For performance and simplicity, we just let map grow to the size of the render chunk.
     * 
     * 3) Mojang only uses the cache for Ao.  Here it is used for all brightness
     * lookups, including flat lighting.
     * 
     * 4) The Mojang cache is a separate threadlocal with a threadlocal boolean to 
     * enable disable. Cache clearing happens with the disable. There's no use case for 
     * us when the cache needs to be disabled (and no apparent case in Mojang's code either)
     * so we simply clear the cache at the start of each new chunk. It is also
     * not a threadlocal because it's held within a threadlocal BlockRenderer.
     */
    private final Long2IntOpenHashMap brightnessCache;
    private final Long2FloatOpenHashMap aoLevelCache;
    
    private final BlockRenderInfo blockInfo;
    class_842 chunkTask; 
    class_849 chunkData;
    class_851 chunkRenderer;
    class_1920 blockView;
    boolean [] resultFlags;
    
    private final AccessBufferBuilder[] buffers = new AccessBufferBuilder[4];
    private final class_1921[] LAYERS = class_1921.values();
    
    private double chunkOffsetX;
    private double chunkOffsetY;
    private double chunkOffsetZ;
    
    // chunk offset + block pos offset + model offsets for plants, etc.
    private float offsetX = 0;
    private float offsetY = 0;
    private float offsetZ = 0;
    
    ChunkRenderInfo(BlockRenderInfo blockInfo) {
        this.blockInfo = blockInfo;
        brightnessCache = new Long2IntOpenHashMap();
        brightnessCache.defaultReturnValue(Integer.MAX_VALUE);
        aoLevelCache = new Long2FloatOpenHashMap();
        aoLevelCache.defaultReturnValue(Float.MAX_VALUE);
    }
    
    void setBlockView(class_853 blockView) {
        this.blockView = blockView;
    }
    
    void setChunkTask(class_842 chunkTask) {
        this.chunkTask = chunkTask;
    }
    
    void prepare(class_851 chunkRenderer, class_2338.class_2339 chunkOrigin, boolean [] resultFlags) {
        this.chunkData = chunkTask.method_3609();
        this.chunkRenderer = chunkRenderer;
        this.resultFlags = resultFlags;
        buffers[0] = null;
        buffers[1] = null;
        buffers[2] = null;
        buffers[3] = null;
        chunkOffsetX = -chunkOrigin.method_10263();
        chunkOffsetY = -chunkOrigin.method_10264();
        chunkOffsetZ = -chunkOrigin.method_10260();
        brightnessCache.clear();
        aoLevelCache.clear();
    }
    
    void release() {
        chunkData = null;
        chunkTask = null;
        chunkRenderer = null;
        buffers[0] = null;
        buffers[1] = null;
        buffers[2] = null;
        buffers[3] = null;
    }
    
    void beginBlock() {
        final class_2680 blockState = blockInfo.blockState;
        final class_2338 blockPos = blockInfo.blockPos;
        offsetX = (float) (chunkOffsetX + blockPos.method_10263());
        offsetY = (float) (chunkOffsetY + blockPos.method_10264());
        offsetZ = (float) (chunkOffsetZ + blockPos.method_10260());

        if(blockState.method_11614().method_16841() != class_2250.field_10656) {
            class_243 offset = blockState.method_11599(blockInfo.blockView, blockPos);
            offsetX += (float)offset.field_1352;
            offsetY += (float)offset.field_1351;
            offsetZ += (float)offset.field_1350;
        }
    }
    
    /** Lazily retrieves output buffer for given layer, initializing as needed. */
    public AccessBufferBuilder getInitializedBuffer(int layerIndex) {
        // redundant for first layer, but probably not faster to check
        resultFlags[layerIndex] = true;
        
        AccessBufferBuilder result = buffers[layerIndex];
        if (result == null) {
            class_287 builder = chunkTask.method_3600().method_3155(layerIndex);
            buffers[layerIndex] = (AccessBufferBuilder) builder;
            class_1921 layer = LAYERS[layerIndex];
            if (!chunkData.method_3649(layer)) {
                chunkData.method_3647(layer); // start buffer
                ((AccessChunkRenderer) chunkRenderer).fabric_beginBufferBuilding(builder, blockInfo.blockPos);
            }
            result = (AccessBufferBuilder) builder;
        }
        return result;
    }
    
    /**
     * Applies position offset for chunk and, if present, block random offset.
     */
    void applyOffsets(MutableQuadViewImpl q) {
        for(int i = 0; i < 4; i++) {
            q.pos(i, q.x(i) + offsetX, q.y(i) + offsetY, q.z(i) + offsetZ);
        }
    }
    
    /**
     * Cached values for {@link BlockState#getBlockBrightness(ExtendedBlockView, BlockPos)}.
     * See also the comments for {@link #brightnessCache}.
     */
    int cachedBrightness(class_2680 blockState, class_2338 pos) {
        long key = pos.method_10063();
        int result = brightnessCache.get(key);
        if (result == Integer.MAX_VALUE) {
            result = blockState.method_11632(blockView, pos);
            brightnessCache.put(key, result);
        }
        return result;
    }
    
    float cachedAoLevel(class_2338 pos)
    {
        long key = pos.method_10063();
        float result = aoLevelCache.get(key);
        if (result == Float.MAX_VALUE) {
            result = blockView.method_8320(pos).method_11596(blockView, pos);
            aoLevelCache.put(key, result);
        }
        return result;
    }
}
