/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.indigo.renderer.render;

import java.util.List;
import java.util.Random;
import java.util.function.Consumer;
import java.util.function.Supplier;

import org.lwjgl.opengl.GL11;

import com.mojang.blaze3d.platform.GlStateManager;

import net.fabricmc.fabric.api.renderer.v1.mesh.Mesh;
import net.fabricmc.fabric.api.renderer.v1.mesh.QuadEmitter;
import net.fabricmc.fabric.api.renderer.v1.model.FabricBakedModel;
import net.fabricmc.fabric.api.renderer.v1.model.ModelHelper;
import net.fabricmc.fabric.api.renderer.v1.render.RenderContext;
import net.fabricmc.indigo.renderer.RenderMaterialImpl;
import net.fabricmc.indigo.renderer.accessor.AccessBufferBuilder;
import net.fabricmc.indigo.renderer.helper.ColorHelper;
import net.fabricmc.indigo.renderer.mesh.EncodingFormat;
import net.fabricmc.indigo.renderer.mesh.MeshImpl;
import net.fabricmc.indigo.renderer.mesh.MutableQuadViewImpl;
import net.minecraft.class_1087;
import net.minecraft.class_1799;
import net.minecraft.class_2680;
import net.minecraft.class_287;
import net.minecraft.class_289;
import net.minecraft.class_290;
import net.minecraft.class_325;
import net.minecraft.class_777;

/**
 * The render context used for item rendering. 
 * Does not implement emissive lighting for sake
 * of simplicity in the default renderer. 
 */
public class ItemRenderContext extends AbstractRenderContext implements RenderContext {
    /** used to accept a method reference from the ItemRenderer */
    @FunctionalInterface
    public static interface VanillaQuadHandler {
        void accept(class_287 bufferBuilder, List<class_777> quads, int color, class_1799 stack);
    }

    private final class_325 colorMap;
    private final Random random = new Random();
    class_287 bufferBuilder;
    AccessBufferBuilder fabricBuffer;
    private int color;
    private class_1799 itemStack;
    private VanillaQuadHandler vanillaHandler;
    private boolean smoothShading = false;
    private boolean enchantment = false;
    
    private final Supplier<Random> randomSupplier = () -> {
        Random result = random;
        result.setSeed(42L);
        return random;
    };
    
    /** 
     * When rendering an enchanted item, input stack will be empty.
     * This value is populated earlier in the call tree when this is the case
     * so that we can render correct geometry and only a single texture.
     */
    public class_1799 enchantmentStack;
    
    private final int[] quadData = new int[EncodingFormat.MAX_STRIDE];;
    
    public ItemRenderContext(class_325 colorMap) {
        this.colorMap = colorMap;
    }
    
    public void renderModel(FabricBakedModel model, int color, class_1799 stack, VanillaQuadHandler vanillaHandler) {
        this.color = color;
        
        if(stack.method_7960() && enchantmentStack != null) {
            enchantment = true;
            this.itemStack = enchantmentStack;
            enchantmentStack = null;
        } else {
            enchantment = false;
            this.itemStack = stack;
        }
        
        this.vanillaHandler = vanillaHandler;
        class_289 tessellator = class_289.method_1348();
        bufferBuilder = tessellator.method_1349();
        fabricBuffer = (AccessBufferBuilder)this.bufferBuilder;
        
        bufferBuilder.method_1328(7, class_290.field_1590);
        model.emitItemQuads(stack, randomSupplier, this);
        tessellator.method_1350();
        
        if(smoothShading) {
            GlStateManager.shadeModel(GL11.GL_FLAT);
            smoothShading = false;
        }
        
        bufferBuilder = null;
        fabricBuffer = null;
        tessellator = null;
        this.itemStack = null;
        this.vanillaHandler = null;
    }

    private class Maker extends MutableQuadViewImpl implements QuadEmitter {
        {
            data = quadData;
            clear();
        }
        
        @Override
        public Maker emit() {
            renderQuad();
            clear();
            return this;
        }
    }
    
    private final Maker editorQuad = new Maker();
    
    private final Consumer<Mesh> meshConsumer = (mesh) -> {
        MeshImpl m = (MeshImpl)mesh;
        final int[] data = m.data();
        final int limit = data.length;
        int index = 0;
        while(index < limit) {
            RenderMaterialImpl.Value mat = RenderMaterialImpl.byIndex(data[index]);
            final int stride = EncodingFormat.stride(mat.spriteDepth());
            System.arraycopy(data, index, editorQuad.data(), 0, stride);
            editorQuad.load();
            index += stride;
            renderQuad();
        }
    };
    
    /**
     * Vanilla normally renders items with flat shading - meaning only
     * the last vertex normal is applied for lighting purposes. We 
     * support non-cube vertex normals so we need to change this to smooth
     * for models that use them.  We don't change it unless needed because
     * OpenGL state changes always impose a performance cost and this happens
     * for every item, every frame.
     */
    private void handleShading() {
        if(!smoothShading && editorQuad.hasVertexNormals()) {
            smoothShading = true;
            GlStateManager.shadeModel(GL11.GL_SMOOTH);
        }
    }
    
    private int quadColor() {
        final int colorIndex = editorQuad.colorIndex();
        int quadColor = color;
        if (!enchantment && quadColor == -1 && colorIndex != 1) {
            quadColor = colorMap.method_1704(itemStack, colorIndex);
            quadColor |= -16777216;
         }
        return quadColor;
    }
    
    private void colorizeAndOutput(int quadColor) {
        final MutableQuadViewImpl q = editorQuad;
        for(int i = 0; i < 4; i++) {
            int c = q.spriteColor(i, 0);
            c = ColorHelper.multiplyColor(quadColor, c);
            q.spriteColor(i, 0, ColorHelper.swapRedBlueIfNeeded(c));
        }
        fabricBuffer.fabric_putVanillaData(quadData, EncodingFormat.VERTEX_START_OFFSET);
    }
    
    private void renderQuad() {
        final MutableQuadViewImpl quad = editorQuad;
        if(!transform(editorQuad)) {
            return;
        }
        
        RenderMaterialImpl.Value mat = quad.material();
        final int quadColor = quadColor();
        final int textureCount = mat.spriteDepth();
        
        handleShading();
        
        // A bit of a hack - copy packed normals on top of lightmaps.
        // Violates normal encoding format but the editor content will be discarded
        // and this avoids the step of copying to a separate array.
        quad.copyNormals(quadData, EncodingFormat.VERTEX_START_OFFSET);
        
        colorizeAndOutput(!enchantment && mat.disableColorIndex(0) ? -1 : quadColor);
        
        // no need to render additional textures for enchantment overlay
        if(!enchantment && textureCount > 1) {
            quad.copyColorUV(1, quadData, EncodingFormat.VERTEX_START_OFFSET);
            colorizeAndOutput(mat.disableColorIndex(1) ? -1 : quadColor);
            
            if(textureCount == 3) {
                quad.copyColorUV(2, quadData, EncodingFormat.VERTEX_START_OFFSET);
                colorizeAndOutput(mat.disableColorIndex(2) ? -1 : quadColor);
            }
        }
    }
    
    @Override
    public Consumer<Mesh> meshConsumer() {
        return meshConsumer;
    }

    private final Consumer<class_1087> fallbackConsumer = model -> {
        for(int i = 0; i < 7; i++) {
            random.setSeed(42L);
            vanillaHandler.accept(bufferBuilder, model.method_4707((class_2680)null, ModelHelper.faceFromIndex(i), random), color, itemStack);
         }
    };
    
    @Override
    public Consumer<class_1087> fallbackConsumer() {
        return fallbackConsumer;
    }

    @Override
    public QuadEmitter getEmitter() {
        editorQuad.clear();
        return editorQuad;
    }
}
