/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.indigo.renderer.render;

import java.util.function.Consumer;

import net.fabricmc.fabric.api.renderer.v1.mesh.Mesh;
import net.fabricmc.fabric.api.renderer.v1.mesh.QuadEmitter;
import net.fabricmc.fabric.api.renderer.v1.model.FabricBakedModel;
import net.fabricmc.fabric.api.renderer.v1.render.RenderContext;
import net.fabricmc.indigo.renderer.aocalc.AoCalculator;
import net.minecraft.class_1087;
import net.minecraft.class_128;
import net.minecraft.class_129;
import net.minecraft.class_148;
import net.minecraft.class_2338;
import net.minecraft.class_2680;
import net.minecraft.class_310;
import net.minecraft.class_776;
import net.minecraft.class_842;
import net.minecraft.class_851;
import net.minecraft.class_853;

/**
 * Implementation of {@link RenderContext} used during terrain rendering.
 * Dispatches calls from models during chunk rebuild to the appropriate consumer,
 * and holds/manages all of the state needed by them.
 */
public class TerrainRenderContext extends AbstractRenderContext implements RenderContext {
    public static final ThreadLocal<TerrainRenderContext> POOL = ThreadLocal.withInitial(TerrainRenderContext::new);
    private final TerrainBlockRenderInfo blockInfo = new TerrainBlockRenderInfo();
    private final ChunkRenderInfo chunkInfo = new ChunkRenderInfo(blockInfo);
    private final AoCalculator aoCalc = new AoCalculator(blockInfo, chunkInfo::cachedBrightness, chunkInfo::cachedAoLevel);
    private final TerrainMeshConsumer meshConsumer = new TerrainMeshConsumer(blockInfo, chunkInfo, aoCalc, this::transform);
    private final TerrainFallbackConsumer fallbackConsumer = new TerrainFallbackConsumer(blockInfo, chunkInfo, aoCalc, this::transform);
    private final class_776 blockRenderManager = class_310.method_1551().method_1541();
    
    public void setBlockView(class_853 blockView) {
        blockInfo.setBlockView(blockView);
        chunkInfo.setBlockView(blockView);
    }
    
    public void setChunkTask(class_842 chunkTask) {
        chunkInfo.setChunkTask(chunkTask);
    }
    
    public TerrainRenderContext prepare(class_851 chunkRenderer, class_2338.class_2339 chunkOrigin, boolean [] resultFlags) {
        chunkInfo.prepare(chunkRenderer, chunkOrigin, resultFlags);
        return this;
    }
    
    public void release() {
        chunkInfo.release();
        blockInfo.release();
    }
    
    /** Called from chunk renderer hook. */
    public boolean tesselateBlock(class_2680 blockState, class_2338 blockPos) {
        try {
            final class_1087 model = blockRenderManager.method_3349(blockState);
            aoCalc.clear();
            blockInfo.prepareForBlock(blockState, blockPos, model.method_4708());
            chunkInfo.beginBlock();
            ((FabricBakedModel)model).emitBlockQuads(blockInfo.blockView, blockInfo.blockState, blockInfo.blockPos, blockInfo.randomSupplier, this);
        } catch (Throwable var9) {
           class_128 crashReport_1 = class_128.method_560(var9, "Tesselating block in world - Indigo Renderer");
           class_129 crashReportElement_1 = crashReport_1.method_562("Block being tesselated");
           class_129.method_586(crashReportElement_1, blockPos, blockState);
           throw new class_148(crashReport_1);
        }
        return chunkInfo.resultFlags[blockInfo.defaultLayerIndex];
     }

    @Override
    public Consumer<Mesh> meshConsumer() {
        return meshConsumer;
    }

    @Override
    public Consumer<class_1087> fallbackConsumer() {
        return fallbackConsumer;
    }

    @Override
    public QuadEmitter getEmitter() {
        return meshConsumer.getEmitter();
    }
}
