/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.client.indigo.renderer;

import java.util.List;
import java.util.Map;

import com.llamalad7.mixinextras.sugar.Local;
import org.spongepowered.asm.mixin.Final;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.Redirect;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfoReturnable;
import net.fabricmc.fabric.impl.client.indigo.renderer.accessor.AccessChunkRendererRegion;
import net.fabricmc.fabric.impl.client.indigo.renderer.render.TerrainRenderContext;
import net.minecraft.class_1087;
import net.minecraft.class_11515;
import net.minecraft.class_1920;
import net.minecraft.class_2338;
import net.minecraft.class_2464;
import net.minecraft.class_2680;
import net.minecraft.class_287;
import net.minecraft.class_4076;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_5819;
import net.minecraft.class_750;
import net.minecraft.class_776;
import net.minecraft.class_8251;
import net.minecraft.class_853;
import net.minecraft.class_9810;

/**
 * Implements the main hooks for terrain rendering. Attempts to tread
 * lightly. This means we are deliberately stepping over some minor
 * optimization opportunities.
 *
 * <p>Non-Fabric renderer implementations that are looking to maximize
 * performance will likely take a much more aggressive approach.
 * For that reason, mod authors who want compatibility with advanced
 * renderers will do well to steer clear of chunk rebuild hooks unless
 * they are creating a renderer.
 *
 * <p>These hooks are intended only for the Fabric default renderer and
 * aren't expected to be present when a different renderer is being used.
 * Renderer authors are responsible for creating the hooks they need.
 * (Though they can use these as an example if they wish.)
 */
@Mixin(class_9810.class)
abstract class SectionBuilderMixin {
	@Shadow
	@Final
	private class_776 blockRenderManager;

	@Shadow
	abstract class_287 beginBufferBuilding(Map<class_11515, class_287> builders, class_750 allocatorStorage, class_11515 layer);

	@Inject(method = "build",
			at = @At(value = "INVOKE", target = "Lnet/minecraft/util/math/BlockPos;iterate(Lnet/minecraft/util/math/BlockPos;Lnet/minecraft/util/math/BlockPos;)Ljava/lang/Iterable;"))
	private void hookBuild(class_4076 sectionPos, class_853 region, class_8251 sorter,
						class_750 allocators,
						CallbackInfoReturnable<class_9810.class_9811> cir,
						@Local(ordinal = 0) class_2338 sectionOrigin,
						@Local(ordinal = 0) class_4587 matrixStack,
						@Local(ordinal = 0) Map<class_11515, class_287> builderMap,
						@Local(ordinal = 0) class_5819 random) {
		// hook just before iterating over the render chunk's blocks to capture the buffer builder map
		TerrainRenderContext renderer = TerrainRenderContext.POOL.get();
		renderer.prepare(region, sectionOrigin, matrixStack, random, layer -> beginBufferBuilding(builderMap, allocators, layer));
		((AccessChunkRendererRegion) region).fabric_setRenderer(renderer);
	}

	/**
	 * This is the hook that actually implements the rendering API for terrain rendering.
	 *
	 * <p>It's unusual to have a @Redirect in a Fabric library, but in this case it is our explicit intention that
	 * {@link class_1087#method_68513(class_5819, List)} and
	 * {@link class_776#method_3355(class_2680, class_2338, class_1920, class_4587, class_4588, boolean, List)}
	 * do not execute for models that will be rendered by our renderer. For performance and convenience, just skip the
	 * entire if block.
	 *
	 * <p>Any mod that wants to redirect this specific call is likely also a renderer, in which case this
	 * renderer should not be present, or the mod should probably instead be relying on the renderer API
	 * which was specifically created to provide for enhanced terrain rendering.
	 */
	@Redirect(method = "build", at = @At(value = "INVOKE", target = "net/minecraft/block/BlockState.getRenderType()Lnet/minecraft/block/BlockRenderType;"))
	private class_2464 hookBuildRenderBlock(class_2680 blockState, class_4076 sectionPos, class_853 renderRegion, class_8251 vertexSorter, class_750 allocatorStorage, @Local(ordinal = 2) class_2338 blockPos) {
		class_2464 blockRenderType = blockState.method_26217();

		if (blockRenderType == class_2464.field_11458) {
			class_1087 model = blockRenderManager.method_3349(blockState);
			((AccessChunkRendererRegion) renderRegion).fabric_getRenderer().bufferModel(model, blockState, blockPos);
			return class_2464.field_11455; // Cancel the vanilla logic
		}

		return blockRenderType;
	}

	/**
	 * Release all references. Probably not necessary but would be $#%! to debug if it is.
	 */
	@Inject(method = "build", at = @At(value = "RETURN"))
	private void hookBuildReturn(class_4076 sectionPos, class_853 renderRegion, class_8251 vertexSorter, class_750 allocatorStorage, CallbackInfoReturnable<class_9810.class_9811> cir) {
		((AccessChunkRendererRegion) renderRegion).fabric_getRenderer().release();
		((AccessChunkRendererRegion) renderRegion).fabric_setRenderer(null);
	}
}
