/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.rendering.data.attachment.client;

import java.util.ConcurrentModificationException;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;

import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Shadow;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import net.fabricmc.fabric.api.rendering.data.v1.RenderAttachedBlockView;
import net.fabricmc.fabric.api.rendering.data.v1.RenderAttachmentBlockEntity;
import net.minecraft.class_1937;
import net.minecraft.class_2338;
import net.minecraft.class_2586;
import net.minecraft.class_2818;
import net.minecraft.class_853;

@Mixin(class_853.class)
public abstract class MixinChunkRendererRegion implements RenderAttachedBlockView {
	private Int2ObjectOpenHashMap<Object> fabric_renderDataObjects;

	@Shadow
	protected abstract int getIndex(class_2338 pos);

	@Shadow
	protected abstract int getIndex(int x, int y, int z);

	private static final AtomicInteger ERROR_COUNTER = new AtomicInteger();
	private static final Logger LOGGER = LogManager.getLogger();

	@Inject(at = @At("RETURN"), method = "<init>")
	public void init(class_1937 world, int cxOff, int czOff, class_2818[][] chunks, class_2338 posFrom, class_2338 posTo, CallbackInfo info) {
		// instantiated lazily - avoids allocation for chunks without any data objects - which is most of them!
		Int2ObjectOpenHashMap<Object> map = null;

		for (class_2818[] chunkOuter : chunks) {
			for (class_2818 chunk : chunkOuter) {
				// Hash maps in chunks should generally not be modified outside of client thread
				// but does happen in practice, due to mods or inconsistent vanilla behaviors, causing
				// CMEs when we iterate the map.  (Vanilla does not iterate these maps when it builds
				// the chunk cache and does not suffer from this problem.)
				//
				// We handle this simply by retrying until it works.  Ugly but effective.
				for (;;) {
					try {
						map = mapChunk(chunk, posFrom, posTo, map);
						break;
					} catch (ConcurrentModificationException e) {
						final int count = ERROR_COUNTER.incrementAndGet();

						if (count <= 5) {
							LOGGER.warn("[Render Data Attachment] Encountered CME during render region build. A mod is accessing or changing chunk data outside the main thread. Retrying.", e);

							if (count == 5) {
								LOGGER.info("[Render Data Attachment] Subsequent exceptions will be suppressed.");
							}
						}
					}
				}
			}
		}

		this.fabric_renderDataObjects = map;
	}

	private Int2ObjectOpenHashMap<Object> mapChunk(class_2818 chunk, class_2338 posFrom, class_2338 posTo, Int2ObjectOpenHashMap<Object> map) {
		final int xMin = posFrom.method_10263();
		final int xMax = posTo.method_10263();
		final int zMin = posFrom.method_10260();
		final int zMax = posTo.method_10260();
		final int yMin = posFrom.method_10264();
		final int yMax = posTo.method_10264();

		for (Map.Entry<class_2338, class_2586> entry : chunk.method_12214().entrySet()) {
			final class_2338 entPos = entry.getKey();

			if (entPos.method_10263() >= xMin && entPos.method_10263() <= xMax
					&& entPos.method_10264() >= yMin && entPos.method_10264() <= yMax
					&& entPos.method_10260() >= zMin && entPos.method_10260() <= zMax) {
				final Object o = ((RenderAttachmentBlockEntity) entry.getValue()).getRenderAttachmentData();

				if (o != null) {
					if (map == null) {
						map = new Int2ObjectOpenHashMap<>();
					}

					map.put(getIndex(entPos), o);
				}
			}
		}

		return map;
	}

	@Override
	public Object getBlockEntityRenderAttachment(class_2338 pos) {
		return fabric_renderDataObjects == null ? null : fabric_renderDataObjects.get(getIndex(pos));
	}
}
