/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.resource.conditions.v1;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import net.fabricmc.fabric.impl.resource.conditions.ResourceConditionsImpl;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_3481;
import net.minecraft.class_3486;
import net.minecraft.class_3489;
import net.minecraft.class_3494;
import net.minecraft.class_3518;
import net.minecraft.class_3611;

/**
 * Contains {@link ConditionJsonProvider}s for resource conditions provided by fabric itself.
 */
public final class DefaultResourceConditions {
	private static final class_2960 NOT = new class_2960("fabric:not");
	private static final class_2960 AND = new class_2960("fabric:and");
	private static final class_2960 OR = new class_2960("fabric:or");
	private static final class_2960 ALL_MODS_LOADED = new class_2960("fabric:all_mods_loaded");
	private static final class_2960 ANY_MOD_LOADED = new class_2960("fabric:any_mod_loaded");
	private static final class_2960 BLOCK_TAGS_POPULATED = new class_2960("fabric:block_tags_populated");
	private static final class_2960 FLUID_TAGS_POPULATED = new class_2960("fabric:fluid_tags_populated");
	private static final class_2960 ITEM_TAGS_POPULATED = new class_2960("fabric:item_tags_populated");

	/**
	 * Create a NOT condition: returns true if its child condition is false, and false if its child is true.
	 */
	public static ConditionJsonProvider not(ConditionJsonProvider value) {
		return new ConditionJsonProvider() {
			@Override
			public void writeParameters(JsonObject object) {
				object.add("value", value.toJson());
			}

			@Override
			public class_2960 getConditionId() {
				return NOT;
			}
		};
	}

	/**
	 * Create a condition that returns true if all of its child conditions are true.
	 */
	public static ConditionJsonProvider and(ConditionJsonProvider... values) {
		return ResourceConditionsImpl.array(AND, values);
	}

	/**
	 * Create a condition that returns true if at least one of its child conditions is true.
	 */
	public static ConditionJsonProvider or(ConditionJsonProvider... values) {
		return ResourceConditionsImpl.array(OR, values);
	}

	/**
	 * Create a condition that returns true if all the passed mod ids correspond to a loaded mod.
	 */
	public static ConditionJsonProvider allModsLoaded(String... modIds) {
		return ResourceConditionsImpl.mods(ALL_MODS_LOADED, modIds);
	}

	/**
	 * Create a condition that returns true if at least one of the passed mod ids corresponds to a loaded mod.
	 */
	public static ConditionJsonProvider anyModLoaded(String... modIds) {
		return ResourceConditionsImpl.mods(ANY_MOD_LOADED, modIds);
	}

	/**
	 * Create a condition that returns true if each of the passed block tags exists and has at least one element.
	 */
	public static ConditionJsonProvider blockTagsPopulated(class_3494.class_5123<class_2248>... tags) {
		return ResourceConditionsImpl.tagsPopulated(BLOCK_TAGS_POPULATED, tags);
	}

	/**
	 * Create a condition that returns true if each of the passed fluid tags exists and has at least one element.
	 */
	public static ConditionJsonProvider fluidTagsPopulated(class_3494.class_5123<class_3611>... tags) {
		return ResourceConditionsImpl.tagsPopulated(FLUID_TAGS_POPULATED, tags);
	}

	/**
	 * Create a condition that returns true if each of the passed item tags exists and has at least one element.
	 */
	public static ConditionJsonProvider itemTagsPopulated(class_3494.class_5123<class_1792>... tags) {
		return ResourceConditionsImpl.tagsPopulated(ITEM_TAGS_POPULATED, tags);
	}

	static void init() {
		// init static
	}

	static {
		ResourceConditions.register(NOT, object -> {
			JsonObject condition = class_3518.method_15296(object, "value");
			return !ResourceConditions.conditionMatches(condition);
		});
		ResourceConditions.register(AND, object -> {
			JsonArray array = class_3518.method_15261(object, "values");
			return ResourceConditions.conditionsMatch(array, true);
		});
		ResourceConditions.register(OR, object -> {
			JsonArray array = class_3518.method_15261(object, "values");
			return ResourceConditions.conditionsMatch(array, false);
		});
		ResourceConditions.register(ALL_MODS_LOADED, object -> ResourceConditionsImpl.modsLoadedMatch(object, true));
		ResourceConditions.register(ANY_MOD_LOADED, object -> ResourceConditionsImpl.modsLoadedMatch(object, false));
		ResourceConditions.register(BLOCK_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_3481.method_15073()));
		ResourceConditions.register(FLUID_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_3486.method_34889()));
		ResourceConditions.register(ITEM_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_3489.method_15106()));
	}

	private DefaultResourceConditions() {
	}
}
