/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.resource.conditions.v1;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import net.fabricmc.fabric.impl.resource.conditions.ResourceConditionsImpl;
import net.minecraft.class_1792;
import net.minecraft.class_2248;
import net.minecraft.class_2960;
import net.minecraft.class_3518;
import net.minecraft.class_3611;
import net.minecraft.class_6862;
import net.minecraft.class_7924;

/**
 * Contains {@link ConditionJsonProvider}s for resource conditions provided by fabric itself.
 */
public final class DefaultResourceConditions {
	private static final class_2960 NOT = new class_2960("fabric:not");
	private static final class_2960 AND = new class_2960("fabric:and");
	private static final class_2960 OR = new class_2960("fabric:or");
	private static final class_2960 ALL_MODS_LOADED = new class_2960("fabric:all_mods_loaded");
	private static final class_2960 ANY_MOD_LOADED = new class_2960("fabric:any_mod_loaded");
	private static final class_2960 BLOCK_TAGS_POPULATED = new class_2960("fabric:block_tags_populated");
	private static final class_2960 FLUID_TAGS_POPULATED = new class_2960("fabric:fluid_tags_populated");
	private static final class_2960 ITEM_TAGS_POPULATED = new class_2960("fabric:item_tags_populated");
	private static final class_2960 TAGS_POPULATED = new class_2960("fabric:tags_populated");

	/**
	 * Creates a NOT condition that returns true if its child condition is false, and false if its child is true.
	 *
	 * @apiNote This condition's ID is {@code fabric:not}, and takes one property, {@code value},
	 * which is a condition.
	 */
	public static ConditionJsonProvider not(ConditionJsonProvider value) {
		return new ConditionJsonProvider() {
			@Override
			public void writeParameters(JsonObject object) {
				object.add("value", value.toJson());
			}

			@Override
			public class_2960 getConditionId() {
				return NOT;
			}
		};
	}

	/**
	 * Creates a condition that returns true if all of its child conditions are true.
	 *
	 * @apiNote This condition's ID is {@code fabric:and}, and takes one property, {@code values},
	 * which is an array of conditions.
	 */
	public static ConditionJsonProvider and(ConditionJsonProvider... values) {
		return ResourceConditionsImpl.array(AND, values);
	}

	/**
	 * Creates a condition that returns true if any of its child conditions are true.
	 *
	 * @apiNote This condition's ID is {@code fabric:or}, and takes one property, {@code values},
	 * which is an array of conditions.
	 */
	public static ConditionJsonProvider or(ConditionJsonProvider... values) {
		return ResourceConditionsImpl.array(OR, values);
	}

	/**
	 * Creates a condition that returns true if all the passed mod ids correspond to a loaded mod.
	 *
	 * @apiNote This condition's ID is {@code fabric:all_mods_loaded}, and takes one property,
	 * {@code values}, which is an array of string mod IDs.
	 */
	public static ConditionJsonProvider allModsLoaded(String... modIds) {
		return ResourceConditionsImpl.mods(ALL_MODS_LOADED, modIds);
	}

	/**
	 * Creates a condition that returns true if at least one of the passed mod ids corresponds to a loaded mod.
	 *
	 * @apiNote This condition's ID is {@code fabric:any_mod_loaded}, and takes one property,
	 * {@code values}, which is an array of string mod IDs.
	 */
	public static ConditionJsonProvider anyModLoaded(String... modIds) {
		return ResourceConditionsImpl.mods(ANY_MOD_LOADED, modIds);
	}

	/**
	 * Create a condition that returns true if each of the passed block tags exists and has at least one element.
	 * @deprecated Use {@link #tagsPopulated} instead.
	 */
	@SafeVarargs
	@Deprecated
	public static ConditionJsonProvider blockTagsPopulated(class_6862<class_2248>... tags) {
		return ResourceConditionsImpl.tagsPopulated(BLOCK_TAGS_POPULATED, false, tags);
	}

	/**
	 * Create a condition that returns true if each of the passed fluid tags exists and has at least one element.
	 * @deprecated Use {@link #tagsPopulated} instead.
	 */
	@SafeVarargs
	@Deprecated
	public static ConditionJsonProvider fluidTagsPopulated(class_6862<class_3611>... tags) {
		return ResourceConditionsImpl.tagsPopulated(FLUID_TAGS_POPULATED, false, tags);
	}

	/**
	 * Create a condition that returns true if each of the passed item tags exists and has at least one element.
	 * @deprecated Use {@link #tagsPopulated} instead.
	 */
	@SafeVarargs
	@Deprecated
	public static ConditionJsonProvider itemTagsPopulated(class_6862<class_1792>... tags) {
		return ResourceConditionsImpl.tagsPopulated(ITEM_TAGS_POPULATED, false, tags);
	}

	/**
	 * Creates a condition that returns true if each of the passed tags exists and has at least one element.
	 * This works for any registries, and the registry ID of the tags is serialized to JSON as well as the tags.
	 *
	 * @apiNote This condition's ID is {@code fabric:tags_populated}, and takes up to two properties:
	 * {@code values}, which is an array of string tag IDs, and {@code registry}, which is the ID of
	 * the registry of the tags. If {@code registry} is not provided, it defaults to {@code minecraft:item}.
	 */
	@SafeVarargs
	public static <T> ConditionJsonProvider tagsPopulated(class_6862<T>... tags) {
		return ResourceConditionsImpl.tagsPopulated(TAGS_POPULATED, true, tags);
	}

	static void init() {
		// init static
	}

	static {
		ResourceConditions.register(NOT, object -> {
			JsonObject condition = class_3518.method_15296(object, "value");
			return !ResourceConditions.conditionMatches(condition);
		});
		ResourceConditions.register(AND, object -> {
			JsonArray array = class_3518.method_15261(object, "values");
			return ResourceConditions.conditionsMatch(array, true);
		});
		ResourceConditions.register(OR, object -> {
			JsonArray array = class_3518.method_15261(object, "values");
			return ResourceConditions.conditionsMatch(array, false);
		});
		ResourceConditions.register(ALL_MODS_LOADED, object -> ResourceConditionsImpl.modsLoadedMatch(object, true));
		ResourceConditions.register(ANY_MOD_LOADED, object -> ResourceConditionsImpl.modsLoadedMatch(object, false));
		ResourceConditions.register(BLOCK_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_7924.field_41254));
		ResourceConditions.register(FLUID_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_7924.field_41270));
		ResourceConditions.register(ITEM_TAGS_POPULATED, object -> ResourceConditionsImpl.tagsPopulatedMatch(object, class_7924.field_41197));
		ResourceConditions.register(TAGS_POPULATED, ResourceConditionsImpl::tagsPopulatedMatch);
	}

	private DefaultResourceConditions() {
	}
}
