/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.resource.loader;

import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collection;
import java.util.function.Predicate;
import net.minecraft.class_2960;
import net.minecraft.class_3258;
import net.minecraft.class_3259;
import net.minecraft.class_3262;
import net.minecraft.class_3264;
import net.minecraft.class_3268;
import org.jetbrains.annotations.Nullable;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Overwrite;
import org.spongepowered.asm.mixin.Unique;

/**
 * Make the default resource pack use the MC jar directly instead of the full classpath.
 * This is a major speed improvement, as well as a bugfix (it prevents other mod jars from overriding MC's resources).
 */
@Mixin(class_3268.class)
public abstract class DefaultResourcePackMixin {
	/**
	 * Redirect all resource access to the MC jar zip pack.
	 */
	final class_3262 fabric_mcJarPack = createJarZipPack();

	@Unique
	private class_3262 createJarZipPack() {
		class_3264 type;

		if (getClass().equals(class_3268.class)) {
			// Server pack
			type = class_3264.field_14190;
		} else {
			// Client pack
			type = class_3264.field_14188;
		}

		// Locate MC jar by finding the URL that contains the assets root.
		try {
			URL assetsRootUrl = class_3268.class.getResource("/" + type.method_14413() + "/.mcassetsroot");
			URLConnection connection = assetsRootUrl.openConnection();

			if (connection instanceof JarURLConnection jarURLConnection) {
				return new class_3258(Paths.get(jarURLConnection.getJarFileURL().toURI()).toFile());
			} else {
				// Not a jar, assume it's a regular directory.
				Path rootPath = Paths.get(assetsRootUrl.toURI()).resolve("../..").toAbsolutePath();
				return new class_3259(rootPath.toFile());
			}
		} catch (Exception exception) {
			throw new RuntimeException("Fabric: Failed to locate Minecraft assets root!", exception);
		}
	}

	/**
	 * @author FabricMC
	 * @reason Gets rid of classpath scanning.
	 */
	@Overwrite
	public Collection<class_2960> findResources(class_3264 type, String namespace, String prefix, Predicate<class_2960> predicate) {
		return fabric_mcJarPack.method_14408(type, namespace, prefix, predicate);
	}

	/**
	 * @author FabricMC
	 * @reason Gets rid of classpath scanning.
	 */
	@Overwrite
	public boolean contains(class_3264 type, class_2960 id) {
		return fabric_mcJarPack.method_14411(type, id);
	}

	/**
	 * @author FabricMC
	 * @reason Close the resource pack we redirect resource access to.
	 */
	@Overwrite
	public void close() {
		fabric_mcJarPack.close();
	}

	/**
	 * @author FabricMC
	 * @reason Gets rid of classpath scanning.
	 */
	@Nullable
	@Overwrite
	public InputStream getInputStream(String path) throws IOException {
		return fabric_mcJarPack.method_14410(path);
	}

	/**
	 * @author FabricMC
	 * @reason Gets rid of classpath scanning.
	 */
	@Nullable
	@Overwrite
	public InputStream findInputStream(class_3264 type, class_2960 id) throws IOException {
		return fabric_mcJarPack.method_14405(type, id);
	}
}
