/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.client.fluid;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.lookup.v1.custom.ApiProviderMap;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariantAttributes;
import net.minecraft.class_1058;
import net.minecraft.class_124;
import net.minecraft.class_1836;
import net.minecraft.class_1920;
import net.minecraft.class_2338;
import net.minecraft.class_2378;
import net.minecraft.class_2561;
import net.minecraft.class_310;
import net.minecraft.class_3611;

/**
 * Client-side display of fluid variants.
 *
 * <p><b>Experimental feature</b>, we reserve the right to remove or change it without further notice.
 * The transfer API is a complex addition, and we want to be able to correct possible design mistakes.
 */
@ApiStatus.Experimental
@Environment(EnvType.CLIENT)
public class FluidVariantRendering {
	private static final ApiProviderMap<class_3611, FluidVariantRenderHandler> HANDLERS = ApiProviderMap.create();
	private static final FluidVariantRenderHandler DEFAULT_HANDLER = new FluidVariantRenderHandler() { };

	/**
	 * Register a render handler for the passed fluid.
	 */
	public static void register(class_3611 fluid, FluidVariantRenderHandler handler) {
		if (HANDLERS.putIfAbsent(fluid, handler) != null) {
			throw new IllegalArgumentException("Duplicate handler registration for fluid " + fluid);
		}
	}

	/**
	 * Return the render handler for the passed fluid, if available, and {@code null} otherwise.
	 */
	@Nullable
	public static FluidVariantRenderHandler getHandler(class_3611 fluid) {
		return HANDLERS.get(fluid);
	}

	/**
	 * Return the render handler for the passed fluid, if available, or the default instance otherwise.
	 */
	public static FluidVariantRenderHandler getHandlerOrDefault(class_3611 fluid) {
		FluidVariantRenderHandler handler = HANDLERS.get(fluid);
		return handler == null ? DEFAULT_HANDLER : handler;
	}

	/**
	 * Return the name of the passed fluid variant.
	 * @deprecated use {@link FluidVariantAttributes#getName} instead.
	 */
	@Deprecated(forRemoval = true)
	public static class_2561 getName(FluidVariant fluidVariant) {
		return getHandlerOrDefault(fluidVariant.getFluid()).getName(fluidVariant);
	}

	/**
	 * Return a mutable list: the tooltip for the passed fluid variant, including the name and additional lines if available
	 * and the id of the fluid if advanced tooltips are enabled.
	 *
	 * <p>Compared to {@linkplain #getTooltip(FluidVariant, class_1836) the other overload}, the current tooltip context is automatically used.
	 */
	public static List<class_2561> getTooltip(FluidVariant fluidVariant) {
		return getTooltip(fluidVariant, class_310.method_1551().field_1690.field_1827 ? class_1836.class_1837.field_8935 : class_1836.class_1837.field_8934);
	}

	/**
	 * Return a mutable list: the tooltip for the passed fluid variant, including the name and additional lines if available
	 * and the id of the fluid if advanced tooltips are enabled.
	 */
	public static List<class_2561> getTooltip(FluidVariant fluidVariant, class_1836 context) {
		List<class_2561> tooltip = new ArrayList<>();

		// Name first
		tooltip.add(getName(fluidVariant));

		// Additional tooltip information
		getHandlerOrDefault(fluidVariant.getFluid()).appendTooltip(fluidVariant, tooltip, context);

		// If advanced tooltips are enabled, render the fluid id
		if (context.method_8035()) {
			tooltip.add(class_2561.method_43470(class_2378.field_11154.method_10221(fluidVariant.getFluid()).toString()).method_27692(class_124.field_1063));
		}

		// TODO: consider adding an event to append to tooltips?

		return tooltip;
	}

	/**
	 * Return the still and the flowing sprite that should be used to render the passed fluid variant, or null if they are not available.
	 * The sprites should be rendered using the color returned by {@link #getColor}.
	 *
	 * @see FluidVariantRenderHandler#getSprites
	 */
	@Nullable
	public static class_1058[] getSprites(FluidVariant fluidVariant) {
		return getHandlerOrDefault(fluidVariant.getFluid()).getSprites(fluidVariant);
	}

	/**
	 * Return the still sprite that should be used to render the passed fluid variant, or null if it's not available.
	 * The sprite should be rendered using the color returned by {@link #getColor}.
	 */
	@Nullable
	public static class_1058 getSprite(FluidVariant fluidVariant) {
		class_1058[] sprites = getSprites(fluidVariant);
		return sprites != null ? Objects.requireNonNull(sprites[0]) : null;
	}

	/**
	 * Return the position-independent color that should be used to render {@linkplain #getSprite the sprite} of the passed fluid variant.
	 */
	public static int getColor(FluidVariant fluidVariant) {
		return getColor(fluidVariant, null, null);
	}

	/**
	 * Return the color that should be used when rendering {@linkplain #getSprite the sprite} of the passed fluid variant.
	 *
	 * <p>If the world and the position parameters are null, a position-independent color is returned.
	 * If the world and position parameters are not null, the color may depend on the position.
	 * For example, if world and position are passed, water will use them to return a biome-dependent color.
	 */
	public static int getColor(FluidVariant fluidVariant, @Nullable class_1920 view, @Nullable class_2338 pos) {
		return getHandlerOrDefault(fluidVariant.getFluid()).getColor(fluidVariant, view, pos);
	}

	/**
	 * Return {@code true} if this fluid variant should be rendered as filling tanks from the top.
	 * @deprecated use {@link FluidVariantAttributes#isLighterThanAir} instead.
	 */
	@Deprecated(forRemoval = true)
	public static boolean fillsFromTop(FluidVariant fluidVariant) {
		return getHandlerOrDefault(fluidVariant.getFluid()).fillsFromTop(fluidVariant);
	}
}
