/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.client.fluid;

import java.util.List;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.api.EnvType;
import net.fabricmc.api.Environment;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandler;
import net.fabricmc.fabric.api.client.render.fluid.v1.FluidRenderHandlerRegistry;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariantAttributes;
import net.minecraft.class_1058;
import net.minecraft.class_1836;
import net.minecraft.class_1920;
import net.minecraft.class_2338;
import net.minecraft.class_2561;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariantAttributeHandler;

/**
 * Defines how {@linkplain FluidVariant fluid variants} of a given Fluid should be displayed to clients.
 * Register with {@link FluidVariantRendering#register}.
 *
 * <p><b>Experimental feature</b>, we reserve the right to remove or change it without further notice.
 * The transfer API is a complex addition, and we want to be able to correct possible design mistakes.
 */
@ApiStatus.Experimental
@Environment(EnvType.CLIENT)
public interface FluidVariantRenderHandler {
	/**
	 * @deprecated Implement {@link FluidVariantAttributeHandler#getName} instead.
	 * This function will be removed in a future iteration of the API.
	 */
	@Deprecated(forRemoval = true)
	default class_2561 getName(FluidVariant fluidVariant) {
		return FluidVariantAttributes.getName(fluidVariant);
	}

	/**
	 * Append additional tooltips to the passed list if additional information is contained in the fluid variant.
	 *
	 * <p>The name of the fluid, and its identifier if the tooltip context is advanced, should not be appended.
	 * They are already added by {@link FluidVariantRendering#getTooltip}.
	 */
	default void appendTooltip(FluidVariant fluidVariant, List<class_2561> tooltip, class_1836 tooltipContext) {
	}

	/**
	 * Return an array of size at least 2 containing the sprites that should be used to render the passed fluid variant,
	 * for use in baked models, (block) entity renderers, or user interfaces.
	 * The first sprite in the array is the still sprite, and the second is the flowing sprite.
	 *
	 * <p>Null may be returned if the fluid variant should not be rendered, but if an array is returned it must have at least two entries and
	 * they may not be null.
	 */
	@Nullable
	default class_1058[] getSprites(FluidVariant fluidVariant) {
		// Use the fluid render handler by default.
		FluidRenderHandler fluidRenderHandler = FluidRenderHandlerRegistry.INSTANCE.get(fluidVariant.getFluid());

		if (fluidRenderHandler != null) {
			return fluidRenderHandler.getFluidSprites(null, null, fluidVariant.getFluid().method_15785());
		} else {
			return null;
		}
	}

	/**
	 * @deprecated Use and implement {@linkplain #getSprites(FluidVariant) the other more general overload}.
	 * This one will be removed in a future iteration of the API.
	 */
	@Deprecated(forRemoval = true)
	@Nullable
	default class_1058 getSprite(FluidVariant fluidVariant) {
		class_1058[] sprites = getSprites(fluidVariant);
		return sprites != null ? sprites[0] : null;
	}

	/**
	 * @deprecated Use and implement {@linkplain #getColor(FluidVariant, class_1920, class_2338) the other more general overload}.
	 * This one will be removed in a future iteration of the API.
	 */
	@Deprecated(forRemoval = true)
	default int getColor(FluidVariant fluidVariant) {
		return getColor(fluidVariant, null, null);
	}

	/**
	 * Return the color to use when rendering {@linkplain #getSprite the sprite} of this fluid variant.
	 * Transparency (alpha) will generally be taken into account and should be specified as well.
	 *
	 * <p>The world and position are optional context parameters and may be {@code null}.
	 * If they are null, this method must return a location-independent color.
	 * If they are provided, this method may return a color that depends on the location.
	 * For example, water returns the biome-dependent color if the context parameters are specified, or its default color if one of them is null.
	 */
	default int getColor(FluidVariant fluidVariant, @Nullable class_1920 view, @Nullable class_2338 pos) {
		// Use the fluid render handler by default.
		FluidRenderHandler fluidRenderHandler = FluidRenderHandlerRegistry.INSTANCE.get(fluidVariant.getFluid());

		if (fluidRenderHandler != null) {
			return fluidRenderHandler.getFluidColor(view, pos, fluidVariant.getFluid().method_15785()) | 255 << 24;
		} else {
			return -1;
		}
	}

	/**
	 * @deprecated Implement {@link FluidVariantAttributeHandler#isLighterThanAir(FluidVariant)} instead.
	 * This function will be removed in a future iteration of the API.
	 */
	@Deprecated(forRemoval = true)
	default boolean fillsFromTop(FluidVariant fluidVariant) {
		// By default, only fluids lighter than air should be filled from top.
		return FluidVariantAttributes.isLighterThanAir(fluidVariant);
	}
}
