/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.item;

import java.util.Objects;

import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_7923;

public class ItemVariantImpl implements ItemVariant {
	public static ItemVariant of(class_1792 item, @Nullable class_2487 tag) {
		Objects.requireNonNull(item, "Item may not be null.");

		// Only tag-less or empty item variants are cached for now.
		if (tag == null || item == class_1802.field_8162) {
			return ((ItemVariantCache) item).fabric_getCachedItemVariant();
		} else {
			return new ItemVariantImpl(item, tag);
		}
	}

	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-transfer-api-v1/item");

	private final class_1792 item;
	private final @Nullable class_2487 nbt;
	private final int hashCode;
	/**
	 * Lazily computed, equivalent to calling toStack(1). <b>MAKE SURE IT IS NEVER MODIFIED!</b>
	 */
	private volatile @Nullable class_1799 cachedStack = null;

	public ItemVariantImpl(class_1792 item, class_2487 nbt) {
		this.item = item;
		this.nbt = nbt == null ? null : nbt.method_10553(); // defensive copy
		hashCode = Objects.hash(item, nbt);
	}

	@Override
	public class_1792 getObject() {
		return item;
	}

	@Nullable
	@Override
	public class_2487 getNbt() {
		return nbt;
	}

	@Override
	public boolean isBlank() {
		return item == class_1802.field_8162;
	}

	@Override
	public class_2487 toNbt() {
		class_2487 result = new class_2487();
		result.method_10582("item", class_7923.field_41178.method_10221(item).toString());

		if (nbt != null) {
			result.method_10566("tag", nbt.method_10553());
		}

		return result;
	}

	public static ItemVariant fromNbt(class_2487 tag) {
		try {
			class_1792 item = class_7923.field_41178.method_10223(new class_2960(tag.method_10558("item")));
			class_2487 aTag = tag.method_10545("tag") ? tag.method_10562("tag") : null;
			return of(item, aTag);
		} catch (RuntimeException runtimeException) {
			LOGGER.debug("Tried to load an invalid ItemVariant from NBT: {}", tag, runtimeException);
			return ItemVariant.blank();
		}
	}

	@Override
	public void toPacket(class_2540 buf) {
		if (isBlank()) {
			buf.writeBoolean(false);
		} else {
			buf.writeBoolean(true);
			buf.method_10804(class_1792.method_7880(item));
			buf.method_10794(nbt);
		}
	}

	public static ItemVariant fromPacket(class_2540 buf) {
		if (!buf.readBoolean()) {
			return ItemVariant.blank();
		} else {
			class_1792 item = class_1792.method_7875(buf.method_10816());
			class_2487 nbt = buf.method_10798();
			return of(item, nbt);
		}
	}

	@Override
	public String toString() {
		return "ItemVariantImpl{item=" + item + ", tag=" + nbt + '}';
	}

	@Override
	public boolean equals(Object o) {
		// succeed fast with == check
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		ItemVariantImpl ItemVariant = (ItemVariantImpl) o;
		// fail fast with hash code
		return hashCode == ItemVariant.hashCode && item == ItemVariant.item && nbtMatches(ItemVariant.nbt);
	}

	@Override
	public int hashCode() {
		return hashCode;
	}

	public class_1799 getCachedStack() {
		class_1799 ret = cachedStack;

		if (ret == null) {
			// multiple stacks could be created at the same time by different threads, but that is not an issue
			cachedStack = ret = toStack();
		}

		return ret;
	}
}
