/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.fluid;

import java.util.Objects;

import org.slf4j.LoggerFactory;
import org.slf4j.Logger;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.minecraft.class_2378;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_2960;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

public class FluidVariantImpl implements FluidVariant {
	public static FluidVariant of(class_3611 fluid, @Nullable class_2487 nbt) {
		Objects.requireNonNull(fluid, "Fluid may not be null.");

		if (!fluid.method_15793(fluid.method_15785()) && fluid != class_3612.field_15906) {
			// Note: the empty fluid is not still, that's why we check for it specifically.
			throw new IllegalArgumentException("Fluid may not be flowing.");
		}

		if (nbt == null || fluid == class_3612.field_15906) {
			// Use the cached variant inside the fluid
			return ((FluidVariantCache) fluid).fabric_getCachedFluidVariant();
		} else {
			// TODO explore caching fluid variants for non null tags.
			return new FluidVariantImpl(fluid, nbt);
		}
	}

	private static final Logger LOGGER = LoggerFactory.getLogger("fabric-transfer-api-v1/fluid");

	private final class_3611 fluid;
	private final @Nullable class_2487 nbt;
	private final int hashCode;

	public FluidVariantImpl(class_3611 fluid, class_2487 nbt) {
		this.fluid = fluid;
		this.nbt = nbt == null ? null : nbt.method_10553(); // defensive copy
		this.hashCode = Objects.hash(fluid, nbt);
	}

	@Override
	public boolean isBlank() {
		return fluid == class_3612.field_15906;
	}

	@Override
	public class_3611 getObject() {
		return fluid;
	}

	@Override
	public @Nullable class_2487 getNbt() {
		return nbt;
	}

	@Override
	public class_2487 toNbt() {
		class_2487 result = new class_2487();
		result.method_10582("fluid", class_2378.field_11154.method_10221(fluid).toString());

		if (nbt != null) {
			result.method_10566("tag", nbt.method_10553());
		}

		return result;
	}

	public static FluidVariant fromNbt(class_2487 compound) {
		try {
			class_3611 fluid = class_2378.field_11154.method_10223(new class_2960(compound.method_10558("fluid")));
			class_2487 nbt = compound.method_10545("tag") ? compound.method_10562("tag") : null;
			return of(fluid, nbt);
		} catch (RuntimeException runtimeException) {
			LOGGER.debug("Tried to load an invalid FluidVariant from NBT: {}", compound, runtimeException);
			return FluidVariant.blank();
		}
	}

	@Override
	public void toPacket(class_2540 buf) {
		if (isBlank()) {
			buf.writeBoolean(false);
		} else {
			buf.writeBoolean(true);
			buf.method_10804(class_2378.field_11154.method_10206(fluid));
			buf.method_10794(nbt);
		}
	}

	public static FluidVariant fromPacket(class_2540 buf) {
		if (!buf.readBoolean()) {
			return FluidVariant.blank();
		} else {
			class_3611 fluid = class_2378.field_11154.method_10200(buf.method_10816());
			class_2487 nbt = buf.method_10798();
			return of(fluid, nbt);
		}
	}

	@Override
	public String toString() {
		return "FluidVariantImpl{fluid=" + fluid + ", tag=" + nbt + '}';
	}

	@Override
	public boolean equals(Object o) {
		// succeed fast with == check
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		FluidVariantImpl fluidVariant = (FluidVariantImpl) o;
		// fail fast with hash code
		return hashCode == fluidVariant.hashCode && fluid == fluidVariant.fluid && nbtMatches(fluidVariant.nbt);
	}

	@Override
	public int hashCode() {
		return hashCode;
	}
}
