/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.fluid;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.transfer.v1.storage.TransferVariant;
import net.fabricmc.fabric.impl.transfer.fluid.FluidVariantImpl;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.minecraft.class_3611;
import net.minecraft.class_3612;

/**
 * An immutable association of a still fluid and an optional NBT tag.
 *
 * <p>Do not extend this class. Use {@link #of(class_3611)} and {@link #of(class_3611, class_2487)} to create instances.
 *
 * <p>{@link net.fabricmc.fabric.api.transfer.v1.client.fluid.FluidVariantRendering} can be used for client-side rendering of fluid variants.
 *
 * <p><b>Fluid variants must always be compared with {@code equals}, never by reference!</b>
 * {@code hashCode} is guaranteed to be correct and constant time independently of the size of the NBT.
 *
 * <p><b>Experimental feature</b>, we reserve the right to remove or change it without further notice.
 * The transfer API is a complex addition, and we want to be able to correct possible design mistakes.
 */
@ApiStatus.Experimental
@ApiStatus.NonExtendable
public interface FluidVariant extends TransferVariant<class_3611> {
	/**
	 * Retrieve a blank FluidVariant.
	 */
	static FluidVariant blank() {
		return of(class_3612.field_15906);
	}

	/**
	 * Retrieve a FluidVariant with a fluid, and a {@code null} tag.
	 */
	static FluidVariant of(class_3611 fluid) {
		return of(fluid, null);
	}

	/**
	 * Retrieve a FluidVariant with a fluid, and an optional tag.
	 */
	static FluidVariant of(class_3611 fluid, @Nullable class_2487 nbt) {
		return FluidVariantImpl.of(fluid, nbt);
	}

	/**
	 * Return the fluid of this variant.
	 */
	default class_3611 getFluid() {
		return getObject();
	}

	/**
	 * Deserialize a variant from an NBT compound tag, assuming it was serialized using {@link #toNbt}.
	 *
	 * <p>If an error occurs during deserialization, it will be logged with the DEBUG level, and a blank variant will be returned.
	 */
	static FluidVariant fromNbt(class_2487 nbt) {
		return FluidVariantImpl.fromNbt(nbt);
	}

	/**
	 * Read a variant from a packet byte buffer, assuming it was serialized using {@link #toPacket}.
	 */
	static FluidVariant fromPacket(class_2540 buf) {
		return FluidVariantImpl.fromPacket(buf);
	}
}
