/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.item;

import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.impl.transfer.item.ItemVariantImpl;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_1935;
import net.minecraft.class_2487;
import net.minecraft.class_2540;
import net.fabricmc.fabric.api.transfer.v1.storage.TransferVariant;

/**
 * An immutable count-less ItemStack, i.e. an immutable association of an item and an optional NBT compound tag.
 *
 * <p>Do not implement, use the static {@code of(...)} functions instead.
 */
@ApiStatus.NonExtendable
public interface ItemVariant extends TransferVariant<class_1792> {
	/**
	 * Retrieve a blank ItemVariant.
	 */
	static ItemVariant blank() {
		return of(class_1802.field_8162);
	}

	/**
	 * Retrieve an ItemVariant with the item and tag of a stack.
	 */
	static ItemVariant of(class_1799 stack) {
		return of(stack.method_7909(), stack.method_7969());
	}

	/**
	 * Retrieve an ItemVariant with an item and without a tag.
	 */
	static ItemVariant of(class_1935 item) {
		return of(item, null);
	}

	/**
	 * Retrieve an ItemVariant with an item and an optional tag.
	 */
	static ItemVariant of(class_1935 item, @Nullable class_2487 tag) {
		return ItemVariantImpl.of(item.method_8389(), tag);
	}

	/**
	 * Return true if the item and tag of this variant match those of the passed stack, and false otherwise.
	 */
	default boolean matches(class_1799 stack) {
		return isOf(stack.method_7909()) && nbtMatches(stack.method_7969());
	}

	/**
	 * Return the item of this variant.
	 */
	default class_1792 getItem() {
		return getObject();
	}

	/**
	 * Create a new item stack with count 1 from this variant.
	 */
	default class_1799 toStack() {
		return toStack(1);
	}

	/**
	 * Create a new item stack from this variant.
	 *
	 * @param count The count of the returned stack. It may lead to counts higher than maximum stack size.
	 */
	default class_1799 toStack(int count) {
		if (isBlank()) return class_1799.field_8037;
		class_1799 stack = new class_1799(getItem(), count);
		stack.method_7980(copyNbt());
		return stack;
	}

	/**
	 * Deserialize a variant from an NBT compound tag, assuming it was serialized using
	 * {@link #toNbt}. If an error occurs during deserialization, it will be logged
	 * with the DEBUG level, and a blank variant will be returned.
	 */
	static ItemVariant fromNbt(class_2487 nbt) {
		return ItemVariantImpl.fromNbt(nbt);
	}

	/**
	 * Write a variant from a packet byte buffer, assuming it was serialized using
	 * {@link #toPacket}.
	 */
	static ItemVariant fromPacket(class_2540 buf) {
		return ItemVariantImpl.fromPacket(buf);
	}
}
