/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.fluid;

import java.util.Objects;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.transfer.v1.fluid.FluidVariant;
import net.minecraft.class_2960;
import net.minecraft.class_3609;
import net.minecraft.class_3611;
import net.minecraft.class_3612;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_9326;

public class FluidVariantImpl implements FluidVariant {
	public static FluidVariant of(class_3611 fluid, class_9326 components) {
		Objects.requireNonNull(fluid, "Fluid may not be null.");
		Objects.requireNonNull(components, "Components may not be null.");

		if (!fluid.method_15793(fluid.method_15785()) && fluid != class_3612.field_15906) {
			// Note: the empty fluid is not still, that's why we check for it specifically.

			if (fluid instanceof class_3609 flowable) {
				// Normalize FlowableFluids to their still variants.
				fluid = flowable.method_15751();
			} else {
				// If not a FlowableFluid, we don't know how to convert -> crash.
				class_2960 id = class_7923.field_41173.method_10221(fluid);
				throw new IllegalArgumentException("Cannot convert flowing fluid %s (%s) into a still fluid.".formatted(id, fluid));
			}
		}

		if (components.method_57848() || fluid == class_3612.field_15906) {
			// Use the cached variant inside the fluid
			return ((FluidVariantCache) fluid).fabric_getCachedFluidVariant();
		} else {
			// TODO explore caching fluid variants for non null tags.
			return new FluidVariantImpl(fluid, components);
		}
	}

	public static FluidVariant of(class_6880<class_3611> fluid, class_9326 components) {
		return of(fluid.comp_349(), components);
	}

	private final class_3611 fluid;
	private final class_9326 components;
	private final int hashCode;

	public FluidVariantImpl(class_3611 fluid, class_9326 components) {
		this.fluid = fluid;
		this.components = components;
		this.hashCode = Objects.hash(fluid, components);
	}

	@Override
	public boolean isBlank() {
		return fluid == class_3612.field_15906;
	}

	@Override
	public class_3611 getObject() {
		return fluid;
	}

	@Override
	public @Nullable class_9326 getComponents() {
		return components;
	}

	@Override
	public String toString() {
		return "FluidVariant{fluid=" + fluid + ", components=" + components + '}';
	}

	@Override
	public boolean equals(Object o) {
		// succeed fast with == check
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		FluidVariantImpl fluidVariant = (FluidVariantImpl) o;
		// fail fast with hash code
		return hashCode == fluidVariant.hashCode && fluid == fluidVariant.fluid && componentsMatches(fluidVariant.components);
	}

	@Override
	public int hashCode() {
		return hashCode;
	}
}
