/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.item;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import net.fabricmc.fabric.api.transfer.v1.context.ContainerItemContext;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.StoragePreconditions;
import net.fabricmc.fabric.api.transfer.v1.storage.base.CombinedSlottedStorage;
import net.fabricmc.fabric.api.transfer.v1.storage.base.SingleSlotStorage;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.fabricmc.fabric.mixin.transfer.ContainerComponentAccessor;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_9288;
import net.minecraft.class_9326;
import net.minecraft.class_9334;

public class ContainerComponentStorage extends CombinedSlottedStorage<ItemVariant, SingleSlotStorage<ItemVariant>> {
	final ContainerItemContext ctx;
	private final class_1792 originalItem;

	public ContainerComponentStorage(ContainerItemContext ctx, int slots) {
		super(Collections.emptyList());
		this.ctx = ctx;
		this.originalItem = ctx.getItemVariant().getItem();

		List<ContainerSlotWrapper> backingList = new ArrayList<>(slots);

		for (int i = 0; i < slots; i++) {
			backingList.add(new ContainerSlotWrapper(i));
		}

		parts = Collections.unmodifiableList(backingList);
	}

	class_9288 container() {
		return ctx.getItemVariant().getComponentMap().method_57830(class_9334.field_49622, class_9288.field_49334);
	}

	ContainerComponentAccessor containerAccessor() {
		return (ContainerComponentAccessor) (Object) container();
	}

	private boolean isStillValid() {
		return ctx.getItemVariant().getItem() == originalItem;
	}

	private class ContainerSlotWrapper implements SingleSlotStorage<ItemVariant> {
		final int slot;

		ContainerSlotWrapper(int slot) {
			this.slot = slot;
		}

		private class_1799 getStack() {
			List<class_1799> stacks = ContainerComponentStorage.this.containerAccessor().fabric_getStacks();

			if (stacks.size() <= slot) return class_1799.field_8037;

			return stacks.get(slot);
		}

		protected boolean setStack(class_1799 stack, TransactionContext transaction) {
			List<class_1799> stacks = ContainerComponentStorage.this.container().method_57489().collect(Collectors.toList());

			while (stacks.size() <= slot) stacks.add(class_1799.field_8037);

			stacks.set(slot, stack);

			ContainerItemContext ctx = ContainerComponentStorage.this.ctx;

			ItemVariant newVariant = ctx.getItemVariant().withComponentChanges(class_9326.method_57841()
							.method_57854(class_9334.field_49622, class_9288.method_57493(stacks))
							.method_57852());

			return ctx.exchange(newVariant, 1, transaction) == 1;
		}

		@Override
		public long insert(ItemVariant insertedVariant, long maxAmount, TransactionContext transaction) {
			StoragePreconditions.notBlankNotNegative(insertedVariant, maxAmount);

			if (!ContainerComponentStorage.this.isStillValid()) return 0;

			class_1799 currentStack = getStack();

			if ((insertedVariant.matches(currentStack) || currentStack.method_7960()) && insertedVariant.getItem().method_31568()) {
				int insertedAmount = (int) Math.min(maxAmount, getCapacity() - currentStack.method_7947());

				if (insertedAmount > 0) {
					currentStack = getStack().method_7972();

					if (currentStack.method_7960()) {
						currentStack = insertedVariant.toStack(insertedAmount);
					} else {
						currentStack.method_7933(insertedAmount);
					}

					if (!setStack(currentStack, transaction)) return 0;

					return insertedAmount;
				}
			}

			return 0;
		}

		@Override
		public long extract(ItemVariant variant, long maxAmount, TransactionContext transaction) {
			StoragePreconditions.notBlankNotNegative(variant, maxAmount);

			if (!ContainerComponentStorage.this.isStillValid()) return 0;

			class_1799 currentStack = getStack();

			if (variant.matches(currentStack)) {
				int extracted = (int) Math.min(currentStack.method_7947(), maxAmount);

				if (extracted > 0) {
					currentStack = getStack().method_7972();
					currentStack.method_7934(extracted);

					if (!setStack(currentStack, transaction)) return 0;

					return extracted;
				}
			}

			return 0;
		}

		@Override
		public boolean isResourceBlank() {
			return getStack().method_7960();
		}

		@Override
		public ItemVariant getResource() {
			return ItemVariant.of(getStack());
		}

		@Override
		public long getAmount() {
			return getStack().method_7947();
		}

		@Override
		public long getCapacity() {
			return getStack().method_7909().method_7882();
		}

		@Override
		public String toString() {
			return "ContainerSlotWrapper[%s#%d]".formatted(ContainerComponentStorage.this.ctx.getItemVariant(), slot);
		}
	}
}
