/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.transfer.v1.item.base;

import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.api.transfer.v1.storage.StoragePreconditions;
import net.fabricmc.fabric.api.transfer.v1.storage.base.SingleSlotStorage;
import net.fabricmc.fabric.api.transfer.v1.transaction.TransactionContext;
import net.fabricmc.fabric.api.transfer.v1.transaction.base.SnapshotParticipant;
import net.fabricmc.fabric.impl.transfer.item.ItemVariantImpl;
import net.minecraft.class_1799;

/**
 * An item variant storage backed by an {@link class_1799}.
 * Implementors should at least override {@link #getStack} and {@link #setStack},
 * and probably {@link #onFinalCommit} as well for {@code markDirty()} and similar calls.
 *
 * <p>{@link #canInsert} and {@link #canExtract} can be used for more precise control over which items may be inserted or extracted.
 * If one of these two functions is overridden to always return false, implementors may also wish to override
 * {@link #supportsInsertion} and/or {@link #supportsExtraction}.
 * {@link #getCapacity(ItemVariant)} can be overridden to change the maximum capacity depending on the item variant.
 */
public abstract class SingleStackStorage extends SnapshotParticipant<class_1799> implements SingleSlotStorage<ItemVariant> {
	/**
	 * Return the stack of this storage. It will be modified directly sometimes to avoid needless copies.
	 * However, any mutation of the stack will directly be followed by a call to {@link #setStack}.
	 * This means that either returning the backing stack directly or a copy is safe.
	 *
	 * @return The current stack.
	 */
	protected abstract class_1799 getStack();

	/**
	 * Set the stack of this storage.
	 */
	protected abstract void setStack(class_1799 stack);

	/**
	 * Return {@code true} if the passed non-blank item variant can be inserted, {@code false} otherwise.
	 */
	protected boolean canInsert(ItemVariant itemVariant) {
		return true;
	}

	/**
	 * Return {@code true} if the passed non-blank item variant can be extracted, {@code false} otherwise.
	 */
	protected boolean canExtract(ItemVariant itemVariant) {
		return true;
	}

	/**
	 * Return the maximum capacity of this storage for the passed item variant.
	 * If the passed item variant is blank, an estimate should be returned.
	 *
	 * <p>If the capacity should be limited by the max count of the item, this function must take it into account.
	 * For example, a storage with a maximum count of 4, or less for items that have a smaller max count,
	 * should override this to return {@code Math.min(super.getCapacity(itemVariant), 4);}.
	 *
	 * @return The maximum capacity of this storage for the passed item variant.
	 */
	protected int getCapacity(ItemVariant itemVariant) {
		return ItemVariantImpl.getMaxStackSize(itemVariant);
	}

	@Override
	public boolean isResourceBlank() {
		return getStack().method_7960();
	}

	@Override
	public ItemVariant getResource() {
		return ItemVariant.of(getStack());
	}

	@Override
	public long getAmount() {
		return getStack().method_7947();
	}

	@Override
	public long getCapacity() {
		return getCapacity(getResource());
	}

	@Override
	public long insert(ItemVariant insertedVariant, long maxAmount, TransactionContext transaction) {
		StoragePreconditions.notBlankNotNegative(insertedVariant, maxAmount);

		class_1799 currentStack = getStack();

		if ((insertedVariant.matches(currentStack) || currentStack.method_7960()) && canInsert(insertedVariant)) {
			int insertedAmount = (int) Math.min(maxAmount, getCapacity(insertedVariant) - currentStack.method_7947());

			if (insertedAmount > 0) {
				updateSnapshots(transaction);
				currentStack = getStack();

				if (currentStack.method_7960()) {
					currentStack = insertedVariant.toStack(insertedAmount);
				} else {
					currentStack.method_7933(insertedAmount);
				}

				setStack(currentStack);

				return insertedAmount;
			}
		}

		return 0;
	}

	@Override
	public long extract(ItemVariant variant, long maxAmount, TransactionContext transaction) {
		StoragePreconditions.notBlankNotNegative(variant, maxAmount);

		class_1799 currentStack = getStack();

		if (variant.matches(currentStack) && canExtract(variant)) {
			int extracted = (int) Math.min(currentStack.method_7947(), maxAmount);

			if (extracted > 0) {
				this.updateSnapshots(transaction);
				currentStack = getStack();
				currentStack.method_7934(extracted);
				setStack(currentStack);

				return extracted;
			}
		}

		return 0;
	}

	@Override
	protected class_1799 createSnapshot() {
		class_1799 original = getStack();
		setStack(original.method_7972());
		return original;
	}

	@Override
	protected void readSnapshot(class_1799 snapshot) {
		setStack(snapshot);
	}

	@Override
	public String toString() {
		return "SingleStackStorage[" + getStack() + "]";
	}
}
