/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.transfer.item;

import java.util.Objects;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.transfer.v1.item.ItemVariant;
import net.fabricmc.fabric.impl.transfer.TransferApiImpl;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1802;
import net.minecraft.class_6880;
import net.minecraft.class_9323;
import net.minecraft.class_9326;
import net.minecraft.class_9334;

public class ItemVariantImpl implements ItemVariant {
	public static ItemVariant of(class_1792 item, class_9326 components) {
		Objects.requireNonNull(item, "Item may not be null.");
		Objects.requireNonNull(components, "Components may not be null.");

		// Only tag-less or empty item variants are cached for now.
		if (components.method_57848() || item == class_1802.field_8162) {
			return ((ItemVariantCache) item).fabric_getCachedItemVariant();
		} else {
			return new ItemVariantImpl(item, components);
		}
	}

	public static ItemVariant of(class_6880<class_1792> item, class_9326 components) {
		return of(item.comp_349(), components);
	}

	private final class_1792 item;
	private final class_9326 components;
	private final int hashCode;
	/**
	 * Lazily computed, equivalent to calling toStack(1). <b>MAKE SURE IT IS NEVER MODIFIED!</b>
	 */
	private volatile @Nullable class_1799 cachedStack = null;

	public ItemVariantImpl(class_1792 item, class_9326 components) {
		this.item = item;
		this.components = components;
		hashCode = Objects.hash(item, components);
	}

	@Override
	public class_1792 getObject() {
		return item;
	}

	@Nullable
	@Override
	public class_9326 getComponents() {
		return components;
	}

	@Override
	public class_9323 getComponentMap() {
		return getCachedStack().method_57353();
	}

	@Override
	public ItemVariant withComponentChanges(class_9326 changes) {
		return of(item, TransferApiImpl.mergeChanges(getComponents(), changes));
	}

	@Override
	public boolean isBlank() {
		return item == class_1802.field_8162;
	}

	@Override
	public String toString() {
		return "ItemVariant{item=" + item + ", components=" + components + '}';
	}

	@Override
	public boolean equals(Object o) {
		// succeed fast with == check
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		ItemVariantImpl ItemVariant = (ItemVariantImpl) o;
		// fail fast with hash code
		return hashCode == ItemVariant.hashCode && item == ItemVariant.item && componentsMatch(ItemVariant.components);
	}

	@Override
	public int hashCode() {
		return hashCode;
	}

	/**
	 * Return the max stack size for this variant, respecting component overrides such as
	 * {@link class_9334#field_50071}.
	 */
	public static int getMaxStackSize(ItemVariant variant) {
		return variant.getComponentMap().method_57830(class_9334.field_50071, variant.getItem().method_7882());
	}

	public class_1799 getCachedStack() {
		class_1799 ret = cachedStack;

		if (ret == null) {
			// multiple stacks could be created at the same time by different threads, but that is not an issue
			cachedStack = ret = toStack();
		}

		return ret;
	}
}
