/*
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

package org.cadixdev.mercury.mixin.annotation;

import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IAnnotationBinding;
import org.cadixdev.mercury.shadow.org.eclipse.jdt.core.dom.IMemberValuePairBinding;

import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A container for data held in the {@code @At} annotation.
 *
 * @author Jadon Fowler
 * @since 0.1.0
 */
public class AtData {
    private static final Pattern DOT_REF_PATTERN = Pattern.compile("([\\w_$/]+)\\.(.*\\(.*?\\).+)");
    private static final Pattern METHOD_REF_PATTERN = Pattern.compile("^(\\(.*?\\).+)$");

    // @At(value = "", target = "")
    public static AtData from(final IAnnotationBinding binding) {
        String injectionPoint = null;
        String combined = null;
        DescData desc = null;

        for (final IMemberValuePairBinding pair : binding.getDeclaredMemberValuePairs()) {
            if (Objects.equals("value", pair.getName())) {
                injectionPoint = (String) pair.getValue();
            }
            else if (Objects.equals("target", pair.getName())) {
                combined = (String) pair.getValue();
            }
            else if ("desc".equals(pair.getName())) {
                desc = DescData.from((IAnnotationBinding) pair.getValue());
            }
        }

        return from(injectionPoint, combined, desc);
    }

    // TODO: Make this not necessary for MixinExtras expression definitions,
    //  we don't need the injection points or descs there, as it's irrelevant.
    //  Realistically the best approach here would be to make InjectTarget carry the target class.
    //  At only needs to carry the injection point on its own otherwise, if it carries any significance.
    /**
     * @param injectionPoint value field
     * @param combined target field
     * */
    public static AtData from(final String injectionPoint, final String combined, final DescData desc) {
        if (combined == null) {
            return new AtData(injectionPoint, null, null, desc);
        }

        String className;
        InjectTarget target;

        Matcher methodDescMatcher = METHOD_REF_PATTERN.matcher(combined);
        if (methodDescMatcher.matches()) {
            return new AtData(injectionPoint, null, InjectTarget.of(combined), desc);
        }

        final int semiIndex = combined.indexOf(';');
        if (semiIndex >= 0) {
            className = combined.substring(1, semiIndex);
            target = InjectTarget.of(combined.substring(semiIndex + 1));
        }
        else {
            Matcher matcher = DOT_REF_PATTERN.matcher(combined);
            if (matcher.matches()) {
                className = matcher.group(1);
                target = InjectTarget.of(matcher.group(2));
            } else {
                // it's just the class name, probably a NEW
                className = combined;
                target = null;
            }
        }

        return new AtData(injectionPoint, className, target, desc);
    }

    private final String injectionPoint;
    private final String className;
    private final InjectTarget target;
    private final DescData desc;

    public AtData(final String injectionPoint, final String className, final InjectTarget target, final DescData desc) {
        this.injectionPoint = injectionPoint;
        this.className = className;
        this.target = target;
        this.desc = desc;
    }

    public String getInjectionPoint() {
        return this.injectionPoint;
    }

    public Optional<String> getClassName() {
        return Optional.ofNullable(this.className);
    }

    public Optional<InjectTarget> getTarget() {
        return Optional.ofNullable(this.target);
    }

    public Optional<DescData> getDesc() {
        return Optional.ofNullable(this.desc);
    }

    @Override
    public String toString() {
        return "AtData{" +
                "injectionPoint='" + this.injectionPoint + '\'' +
                ", className='" + this.className + '\'' +
                ", target=" + this.target +
                '}';
    }

}
